// ui_cursor.js
// Purpose: isolate cursor readout wiring (scope-like) so ui.js stays slimmer.
// Behavior/numerics intentionally unchanged.

import { $, clamp, fmt } from "./dom.js";
import { Rtotal } from "./core/derived.js";

export function bindCursorReadout(opts){
  const {
    scopeCanvas,
    getLastScopeData,
    setCursor,
    clearCursor,
    setCursorSample,
    clearCursorSample
  } = opts || {};

  const box = $("#cursorBody");
  if (!scopeCanvas || !box || typeof getLastScopeData !== "function") return;

  function fmtTime(t) {
    const at = Math.abs(t);
    if (at < 1e-6) return `${(t * 1e9).toFixed(1)} ns`;
    if (at < 1e-3) return `${(t * 1e6).toFixed(2)} µs`;
    if (at < 1)    return `${(t * 1e3).toFixed(3)} ms`;
    return `${t.toFixed(6)} s`;
  }

  scopeCanvas.addEventListener("mousemove", (ev) => {
    const d = getLastScopeData();
    if (!d) return;

    const rect = scopeCanvas.getBoundingClientRect();
    const x = ev.clientX - rect.left;
    const y = ev.clientY - rect.top;

    // Always report cursor x to renderer (line), if provided
    if (typeof setCursor === "function") setCursor(x, false);

    // Frame layout:
    // d.geom.{x0,y0,w,h}
    // d.time.{tStart,dt}
    // d.waves.{vSrc,i,vL,vC?}
    const g = d.geom;
    const tm = d.time;
    const wv = d.waves;

    // Only active inside plot area
    if (x < g.x0 || x > g.x0 + g.w || y < g.y0 || y > g.y0 + g.h) {
      if (typeof setCursor === "function") setCursor(x, false);
      box.textContent = "Move mouse over plot…";
      if (typeof clearCursorSample === "function") clearCursorSample();
      return;
    }

    if (typeof setCursor === "function") setCursor(x, true);

    const u = clamp((x - g.x0) / g.w, 0, 1);
    const N = wv.vSrc.length;
    const idx = Math.max(0, Math.min(N - 1, Math.round(u * (N - 1))));
    const t = tm.tStart + idx * tm.dt;

    const vsrc = wv.vSrc[idx];
    const i    = wv.i[idx];
    const vL   = wv.vL[idx];
    const vC   = (wv.vC && wv.vC.length) ? wv.vC[idx] : null;
    const vR   = i * Rtotal();

    if (typeof setCursorSample === "function") {
      setCursorSample({ active: true, vsrc, i, vL, vR, vC });
    }

    box.textContent =
      `t     ${fmtTime(t)}\n` +
      `vsrc  ${fmt(vsrc)} V\n` +
      `i     ${fmt(i)} A\n` +
      `vL    ${fmt(vL)} V\n` +
      (vC !== null ? `vC    ${fmt(vC)} V\n` : ``) +
      `vR    ${fmt(vR)} V`;
  });

  scopeCanvas.addEventListener("mouseleave", () => {
    if (typeof clearCursor === "function") clearCursor();
    box.textContent = "Move mouse over plot…";
    if (typeof clearCursorSample === "function") clearCursorSample();
  });
}