// ui.js
// Purpose: ALL UI/event wiring (sliders, tabs, mode switches, ref buttons, cursor readout, timebase).
// This file is intentionally "dev-friendly": it does not render waveforms and does not start the RAF loop.

// Expected imports (adjust paths if your folder differs)
import { $, clamp, logMap, invLogMap, bindPair, flashButton, fmt } from "./dom.js";
import { state } from "./state.js";
import { tau, Rtotal } from "./core/derived.js";
import { powerSplit, impulse_abs_vL_0_to, peak_abs_vL_0_to } from "./metrics.js";
import { resizeXY, clearXY } from "./lissajous.js";

// --- Locale-safe number parsing (allows "." or ",") ---
function parseNum(v){
  return parseFloat(String(v).replace(",", "."));
}

/**
 * Apply mode UI (Pulse vs Sine).
 * Keeps this in ui.js because it is purely DOM visibility + labeling.
 */
export function applyModeUI() {
  const pulseBox = $("#pulseControls");
  const sineBox  = $("#sineControls");
  const lblV     = $("#lblV");

  if (state.srcMode === "sine") {
    if (pulseBox) pulseBox.style.display = "none";
    if (sineBox)  sineBox.style.display  = "";
    if (lblV)     lblV.textContent       = "Amplitude (sine, peak)";
  } else {
    if (pulseBox) pulseBox.style.display = "";
    if (sineBox)  sineBox.style.display  = "none";
    if (lblV)     lblV.textContent       = "Amplitude (pulse)";
  }
}

/**
 * Bind all UI behavior.
 *
 * You must pass a few renderer-related hooks so ui.js does NOT need to import renderer globals
 * (this avoids circular imports and makes it easier to reorganize later).
 *
 * @param {object} opts
 * @param {HTMLCanvasElement} opts.scopeCanvas  - canvas element for cursor readout
 * @param {Function} opts.resizeScope           - function that resizes the scope canvas
 * @param {Function} opts.getLastScopeData      - returns lastScopeData (or null)
 * @param {Function} opts.setCursor             - (xCssPx:number, active:boolean) => void
 * @param {Function} opts.clearCursor           - () => void
 */
export function bindUI(opts) {
  const {
    scopeCanvas,
    resizeScope,
    getLastScopeData,
    setCursor,
    clearCursor,
    setCursorSample,
    clearCursorSample
  } = opts || {};

  // -------- Safety checks (fail silently) --------
  if (typeof resizeScope === "function") {
    resizeScope();
    window.addEventListener("resize", resizeScope);
  }

  // -------------------------
  // Cursor readout (scope-like)
  // -------------------------
  (function bindCursorReadout() {
    const box = $("#cursorBody");
    if (!scopeCanvas || !box || typeof getLastScopeData !== "function") return;

    function fmtTime(t) {
      const at = Math.abs(t);
      if (at < 1e-6) return `${(t * 1e9).toFixed(1)} ns`;
      if (at < 1e-3) return `${(t * 1e6).toFixed(2)} µs`;
      if (at < 1)    return `${(t * 1e3).toFixed(3)} ms`;
      return `${t.toFixed(6)} s`;
    }

    scopeCanvas.addEventListener("mousemove", (ev) => {
      const d = getLastScopeData();
      if (!d) return;

      const rect = scopeCanvas.getBoundingClientRect();
      const x = ev.clientX - rect.left;
      const y = ev.clientY - rect.top;

      // Always report cursor x to renderer (line), if provided
      if (typeof setCursor === "function") setCursor(x, false);

      // Frame layout (new):
      // d.geom.{x0,y0,w,h}
      // d.time.{tStart,dt}
      // d.waves.{vSrc,i,vL}
      const g = d.geom;
      const tm = d.time;
      const wv = d.waves;

      // Only active inside plot area
      if (x < g.x0 || x > g.x0 + g.w || y < g.y0 || y > g.y0 + g.h) {
        if (typeof setCursor === "function") setCursor(x, false);
        box.textContent = "Move mouse over plot…";
        if (typeof clearCursorSample === "function") clearCursorSample();
        return;
      }

      if (typeof setCursor === "function") setCursor(x, true);

      const u = clamp((x - g.x0) / g.w, 0, 1);
      const N = wv.vSrc.length;
      const idx = Math.max(0, Math.min(N - 1, Math.round(u * (N - 1))));
      const t = tm.tStart + idx * tm.dt;

      const vsrc = wv.vSrc[idx];
      const i    = wv.i[idx];
      const vL   = wv.vL[idx];
      const vC   = (wv.vC && wv.vC.length) ? wv.vC[idx] : null;
      const vR   = i * Rtotal();

      if (typeof setCursorSample === "function") {
        setCursorSample({ active: true, vsrc, i, vL, vR, vC });
      }

      box.textContent =
        `t     ${fmtTime(t)}\n` +
        `vsrc  ${fmt(vsrc)} V\n` +
        `i     ${fmt(i)} A\n` +
        `vL    ${fmt(vL)} V\n` +
        (vC !== null ? `vC    ${fmt(vC)} V\n` : ``) +
        `vR    ${fmt(vR)} V`;
    });

    scopeCanvas.addEventListener("mouseleave", () => {
      if (typeof clearCursor === "function") clearCursor();
      box.textContent = "Move mouse over plot…";
      if (typeof clearCursorSample === "function") clearCursorSample();
    });
  })();
  

  // -------------------------
  // Sliders + numeric inputs (bindPair)
  // -------------------------
  function bindUnitPair(rangeSel, numSel, unitSel, getSI, setSI, cfgByUnit = null) {
    const r = $(rangeSel);
    const n = $(numSel);
    const u = $(unitSel);
    if (!r || !n || !u) return;

    const unitFactor = () => {
      const f = parseFloat(u.value);
      return isFinite(f) && f > 0 ? f : 1;
    };

    function applyCfg() {
      if (!cfgByUnit) return;
      const key = String(u.value);
      const cfg = cfgByUnit[key] || cfgByUnit["*"];
      if (!cfg) return;

      if (cfg.min != null)  r.min  = String(cfg.min);
      if (cfg.max != null)  r.max  = String(cfg.max);
      if (cfg.step != null) r.step = String(cfg.step);

      if (cfg.numStep != null) n.step = String(cfg.numStep);
    }

    // SI -> display (both slider + number are display units)
    const syncFromSI = () => {
      const si = getSI();
      const f = unitFactor();
      const disp = si / f;
      r.value = disp;
      n.value = disp;
    };

    // slider -> SI
    r.addEventListener("input", () => {
      const disp = parseNum(r.value);
      if (!isFinite(disp)) return;
      setSI(disp * unitFactor());
      n.value = disp;
    });

    // number -> SI
    n.addEventListener("input", () => {
      const disp = parseNum(n.value);
      if (!isFinite(disp)) return;
      setSI(disp * unitFactor());
      r.value = disp;
    });

    // unit change: keep SI constant, but reconfigure slider feel + redraw display
    u.addEventListener("change", () => {
      applyCfg();
      syncFromSI();
    });

    applyCfg();
    syncFromSI();
  }


  // L in H internally, display in H/mH/µH
  (function bindLLog(){
    const r = $("#L");
    const n = $("#L_num");
    const u = $("#L_unit");
    if(!r || !n || !u) return;

    const L_MIN = 1e-9;
    const L_MAX = 10;
    const SL_MAX = 1000;

    r.min = "0";
    r.max = SL_MAX;
    r.step = "1";

    function fmt(x){
      return Number(x).toPrecision(6).replace(/\.?0+$/,'');
    }

    function setFromSI(si){
      if(!isFinite(si) || si <= 0){
        state.L = 0;
        n.value = "0";
        r.value = "0";
        return;
      }

      si = clamp(si, L_MIN, L_MAX);
      state.L = si;

      const uf = parseFloat(u.value) || 1;
      n.value = fmt(si / uf);

      const pos = invLogMap(si, L_MIN, L_MAX);
      r.value = 1 + Math.round(pos * (SL_MAX - 1));
    }

    // Slider → SI
    r.addEventListener("input", () => {
      const val = parseInt(r.value,10);
      if(val <= 0){ setFromSI(0); return; }
      const pos = clamp((val - 1)/(SL_MAX - 1),0,1);
      setFromSI(logMap(pos, L_MIN, L_MAX));
    });

    // Typing (no overwrite while typing!)
    n.addEventListener("input", () => {
      const disp = parseNum(n.value);
      if(!isFinite(disp)) return;
      const uf = parseFloat(u.value) || 1;
      state.L = disp * uf; // only update state
    });

    // Normalize on blur
    n.addEventListener("blur", () => {
      setFromSI(state.L);
    });

    // Arrow fine tuning
    n.addEventListener("keydown", (e)=>{
      if(e.key !== "ArrowUp" && e.key !== "ArrowDown") return;
      e.preventDefault();

      const uf = parseFloat(u.value) || 1;
      const cur = isFinite(parseNum(n.value)) ? parseNum(n.value) : state.L/uf;

      let step = cur !== 0 ? Math.abs(cur)*0.01 : 1;
      if(e.shiftKey) step *= 10;
      if(e.altKey) step *= 0.1;

      const next = Math.max(0, cur + (e.key==="ArrowUp"?step:-step));
      setFromSI(next * uf);
    });

    u.addEventListener("change", ()=> setFromSI(state.L));
    setFromSI(state.L);
  })();

  // Rcoil in Ω internally, display in Ω / mΩ
  // LOG slider, 0 = OFF, max = 100 Ω
  (function bindRcoilLog(){
    const r = $("#Rcoil");
    const n = $("#Rcoil_num");
    const u = $("#Rcoil_unit");
    if(!r || !n || !u) return;

    const R_MIN = 1e-6;   // 1 µΩ floor (for log math)
    const R_MAX = 100;    // 100 Ω maximum
    const SL_MAX = 1000;

    r.min = "0";
    r.max = String(SL_MAX);
    r.step = "1";

    function unitFactor(){
      const f = parseFloat(u.value);
      return (isFinite(f) && f > 0) ? f : 1;
    }

    function fmtNum(x){
      return Number(x).toPrecision(6).replace(/\.?0+$/,'');
    }

    function setFromSI(si){
      if(!isFinite(si) || si <= 0){
        state.Rcoil = 0;
        n.value = "0";
        r.value = "0";
        return;
      }

      si = clamp(si, R_MIN, R_MAX);
      state.Rcoil = si;

      const uf = unitFactor();
      n.value = fmtNum(si / uf);

      const pos = invLogMap(si, R_MIN, R_MAX);
      r.value = String(1 + Math.round(pos * (SL_MAX - 1)));
    }

    // slider → SI
    r.addEventListener("input", () => {
      const val = parseInt(r.value, 10);
      if(val <= 0){ setFromSI(0); return; }
      const pos = clamp((val - 1)/(SL_MAX - 1), 0, 1);
      const si = logMap(pos, R_MIN, R_MAX);
      setFromSI(si);
    });

    // typing
    n.addEventListener("input", () => {
      const disp = parseNum(n.value);
      if(!isFinite(disp)) return;
      state.Rcoil = disp * unitFactor();
    });

    n.addEventListener("blur", () => {
      setFromSI(state.Rcoil);
    });

    // arrow fine tuning
    n.addEventListener("keydown", (e)=>{
      if(e.key !== "ArrowUp" && e.key !== "ArrowDown") return;
      e.preventDefault();

      const uf = unitFactor();
      const cur = isFinite(parseNum(n.value)) ? parseNum(n.value) : (state.Rcoil/uf);

      let step = (cur !== 0) ? Math.abs(cur) * 0.01 : 0.001;
      if(e.shiftKey) step *= 10;
      if(e.altKey)   step *= 0.1;

      const nextDisp = Math.max(0, cur + (e.key==="ArrowUp" ? step : -step));
      setFromSI(nextDisp * uf);
    });

    u.addEventListener("change", () => setFromSI(state.Rcoil));

    setFromSI(state.Rcoil);
  })();
  
  // C in F internally, display in F/mF/µF/nF
  (function bindCLog(){
    const r = $("#C");
    const n = $("#C_num");
    const u = $("#C_unit");
    if(!r || !n || !u) return;

    const C_MIN = 1e-12;
    const C_MAX = 0.1;
    const SL_MAX = 1000;

    r.min = "0";
    r.max = SL_MAX;
    r.step = "1";

    function fmt(x){
      return Number(x).toPrecision(6).replace(/\.?0+$/,'');
    }

    function setFromSI(si){
      if(!isFinite(si) || si <= 0){
        state.C = 0;
        n.value = "0";
        r.value = "0";
        return;
      }

      si = clamp(si, C_MIN, C_MAX);
      state.C = si;

      const uf = parseFloat(u.value) || 1;
      n.value = fmt(si / uf);

      const pos = invLogMap(si, C_MIN, C_MAX);
      r.value = 1 + Math.round(pos * (SL_MAX - 1));
    }

    // Slider → SI
    r.addEventListener("input", () => {
      const val = parseInt(r.value,10);
      if(val <= 0){ setFromSI(0); return; }
      const pos = clamp((val - 1)/(SL_MAX - 1),0,1);
      setFromSI(logMap(pos, C_MIN, C_MAX));
    });

    // Typing (no overwrite while typing!)
    n.addEventListener("input", () => {
      const disp = parseNum(n.value);
      if(!isFinite(disp)) return;
      const uf = parseFloat(u.value) || 1;
      state.C = disp * uf; // only update state
    });

    // Normalize on blur
    n.addEventListener("blur", () => {
      setFromSI(state.C);
    });

    // Arrow fine tuning
    n.addEventListener("keydown", (e)=>{
      if(e.key !== "ArrowUp" && e.key !== "ArrowDown") return;
      e.preventDefault();

      const uf = parseFloat(u.value) || 1;
      const cur = isFinite(parseNum(n.value)) ? parseNum(n.value) : state.C/uf;

      let step = cur !== 0 ? Math.abs(cur)*0.01 : 1;
      if(e.shiftKey) step *= 10;
      if(e.altKey) step *= 0.1;

      const next = Math.max(0, cur + (e.key==="ArrowUp"?step:-step));
      setFromSI(next * uf);
    });

    u.addEventListener("change", ()=> setFromSI(state.C));
    setFromSI(state.C);
  })();

  // Topology select (seriesRLC vs C_parallel_RL)
  (function bindTopology(){
    const sel = $("#topology");
    if(!sel) return;

    sel.value = state.topology || "seriesRLC";

    sel.addEventListener("change", () => {
      state.topology = sel.value;
      state.refSnapshot = null;
    });
  })();

  // Rload in Ω internally, display in Ω/mΩ, LOG slider with 0 = OFF
  (function bindRloadLog(){
    const r = $("#Rload");       // range: position 0..1000
    const n = $("#Rload_num");   // number: display units
    const u = $("#Rload_unit");  // unit select
    if(!r || !n || !u) return;

    // Internal range (Ω)
    const R_MIN = 1e-6;        // 1 µΩ floor (for log)
    const R_MAX = 5e8;         // 500 MΩ cap (matches your number max)
    const SL_MAX = 1000;       // slider positions

    // Configure slider as position control
    r.min = "0";
    r.max = String(SL_MAX);
    r.step = "1";

    function unitFactor(){
      const f = parseFloat(u.value);
      return (isFinite(f) && f > 0) ? f : 1; // Ω => 1, mΩ => 1e-3
    }

    function fmtNum(x){
      // compact but stable display (no trailing zeros)
      return Number(x).toPrecision(8).replace(/\.?0+$/,'');
    }

    function setFromSI(si){
      // 0 = OFF
      if(!isFinite(si) || si <= 0){
        state.Rload = 0;
        n.value = "0";
        r.value = "0";
        return;
      }

      si = clamp(si, R_MIN, R_MAX);
      state.Rload = si;

      const uf = unitFactor();
      n.value = fmtNum(si / uf);

      const pos = invLogMap(si, R_MIN, R_MAX); // 0..1
      r.value = String(1 + Math.round(pos * (SL_MAX - 1))); // 1..1000
    }

    // Slider -> SI
    r.addEventListener("input", () => {
      const val = parseInt(r.value, 10);
      if(val <= 0){ setFromSI(0); return; }
      const pos = clamp((val - 1) / (SL_MAX - 1), 0, 1);
      const si = logMap(pos, R_MIN, R_MAX);
      setFromSI(si);
    });

    // Typing -> update state only (don’t fight the user while typing)
    n.addEventListener("input", () => {
      const disp = parseNum(n.value);
      if(!isFinite(disp)) return;
      state.Rload = disp * unitFactor();
    });

    // Normalize on blur (snap into range + sync slider)
    n.addEventListener("blur", () => {
      setFromSI(state.Rload);
    });

    // Arrow fine tuning (relative, with SHIFT/ALT)
    n.addEventListener("keydown", (e)=>{
      if(e.key !== "ArrowUp" && e.key !== "ArrowDown") return;
      e.preventDefault();

      const uf = unitFactor();
      const cur = isFinite(parseNum(n.value)) ? parseNum(n.value) : (state.Rload/uf);

      // relative step (1%), but sane minimum
      let step = (cur !== 0) ? Math.abs(cur) * 0.01 : 1;
      if(step < 1e-6/uf) step = 1e-6/uf; // keep > log floor in display units

      if(e.shiftKey) step *= 10;
      if(e.altKey)   step *= 0.1;

      const nextDisp = Math.max(0, cur + (e.key==="ArrowUp" ? step : -step));
      setFromSI(nextDisp * uf);
    });

    // Unit change: keep SI constant, just rescale display + slider
    u.addEventListener("change", () => setFromSI(state.Rload));

    // Init
    setFromSI(state.Rload);
  })();

  bindPair("#V", "#V_num", (v) => { state.V = v; });


// Pulse controls + Sine controls
// -------------------------
// Frequency controls (log-position slider 0..1000 + unit-aware number input)
//
// IMPORTANT:
// - Internal state is always in Hz (state.f / state.fs).
// - The number box is in the currently selected unit (Hz/kHz/MHz).
// - Changing the unit keeps the underlying Hz constant and just rescales the display.
//
// Range: 1 Hz .. 5 MHz (0 Hz is not supported because log mapping requires >0).
const FREQ_MIN_HZ = 1;
const FREQ_MAX_HZ = 5_000_000;

function bindFreqLogControl(sliderSel, numSel, unitSel, getHz, setHz, opts = {}) {
  const r = $(sliderSel);
  const n = $(numSel);
  const u = $(unitSel);
  if (!r || !n || !u) return;

  // Make slider a pure log-position control (matches logMap/invLogMap usage everywhere else)
  r.min = "0";
  r.max = "1000";
  r.step = "1";

  const minHz = opts.minHz ?? FREQ_MIN_HZ;
  const maxHz = opts.maxHz ?? FREQ_MAX_HZ;

  const unitFactor = () => {
    const f = parseFloat(u.value);
    return isFinite(f) && f > 0 ? f : 1;
  };

  function applyNumLimits() {
    const uf = unitFactor();
    n.min = String(minHz / uf);
    n.max = String(maxHz / uf);
    n.step = "any";
  }

  function setFromHz(hz, { fromNumber = false } = {}) {
    if (!isFinite(hz)) return;

    hz = clamp(hz, minHz, maxHz);
    setHz(hz);

    const uf = unitFactor();
    const disp = hz / uf;

    if (!fromNumber) n.value = String(disp);

    const pos = invLogMap(hz, minHz, maxHz);
    r.value = String(Math.round(clamp(pos, 0, 1) * 1000));
  }

  // Slider → Hz (optionally throttled)
  let tmr = null;
  let lastHz = getHz();

  const throttleMs = opts.throttleMs ?? 0;

  r.addEventListener("input", () => {
    const u01 = clamp(parseInt(r.value, 10) / 1000, 0, 1);
    const hz = logMap(u01, minHz, maxHz);

    const uf = unitFactor();
    n.value = String(hz / uf); // instant feedback in current unit

    if (throttleMs > 0) {
      lastHz = hz;
      if (tmr) return;
      tmr = setTimeout(() => {
        tmr = null;
        setFromHz(lastHz);
      }, throttleMs);
    } else {
      setFromHz(hz);
    }
  });

  // Commit final immediately (mouse up / touch end)
  r.addEventListener("change", () => {
    if (tmr) { clearTimeout(tmr); tmr = null; }
    const u01 = clamp(parseInt(r.value, 10) / 1000, 0, 1);
    const hz = logMap(u01, minHz, maxHz);
    setFromHz(hz);
  });

  // Number → Hz (immediate, in selected unit)
  n.addEventListener("input", () => {
    const disp = parseNum(n.value);
    if (!isFinite(disp)) return;
    const hz = disp * unitFactor();
    setFromHz(hz, { fromNumber: true });
  });

  n.addEventListener("blur", () => {
    const disp = parseNum(n.value);
    if (!isFinite(disp)) { setFromHz(getHz()); return; }
    setFromHz(disp * unitFactor());
  });

  // Unit change keeps underlying Hz constant
  u.addEventListener("change", () => {
    applyNumLimits();
    setFromHz(getHz());
  });

  // Init
  applyNumLimits();
  setFromHz(getHz());
}

// Pulse repetition frequency (state.f) — throttled a bit to keep heavy pulse recompute smooth
bindFreqLogControl(
  "#f", "#f_num", "#f_unit",
  () => state.f || FREQ_MIN_HZ,
  (hz) => { state.f = hz; },
  { minHz: FREQ_MIN_HZ, maxHz: FREQ_MAX_HZ, throttleMs: 25 }
);

bindPair("#pw", "#pw_num", (v) => { state.dutyPct = v; });

// Sine frequency (state.fs) — no throttle (lightweight)
bindFreqLogControl(
  "#fs", "#fs_num", "#fs_unit",
  () => state.fs || FREQ_MIN_HZ,
  (hz) => { state.fs = hz; },
  { minHz: FREQ_MIN_HZ, maxHz: FREQ_MAX_HZ, throttleMs: 0 }
);

  // -------------------------
  // Timebase (log slider)
  // -------------------------
  // Keep the same range you used in app.js
  const TB_MIN_MS = 0.002; // 2 µs/div
  const TB_MAX_MS = 200.0; // 200 ms/div

  (function bindTimebaseLog() {
    const r = $("#timeDiv");
    const n = $("#timeDiv_num");
    if (!r || !n) return;

    // Configure slider as position (0..1000)
    r.min = "0";
    r.max = "1000";
    r.step = "1";

    function setFromMs(ms) {
      ms = clamp(ms, TB_MIN_MS, TB_MAX_MS);
      state.timeDiv_ms = ms;

      // numeric input (ms/div)
      n.value = ms;

      // slider position
      const u = invLogMap(ms, TB_MIN_MS, TB_MAX_MS);
      r.value = String(Math.round(clamp(u, 0, 1) * 1000));
    }

    r.addEventListener("input", () => {
      const u = clamp(parseInt(r.value, 10) / 1000, 0, 1);
      const ms = logMap(u, TB_MIN_MS, TB_MAX_MS);
      setFromMs(ms);
    });

    n.addEventListener("input", () => {
      const ms = parseFloat(n.value);
      if (isFinite(ms)) setFromMs(ms);
    });

    // Init from current state
    setFromMs(state.timeDiv_ms);
  })();


  // -------------------------
  // Tabs (Metrics card)
  // -------------------------
  document.querySelectorAll(".tabBtn").forEach((btn) => {
    btn.addEventListener("click", () => {
      const id = btn.dataset.tab;

      document.querySelectorAll(".tabBtn").forEach((b) =>
        b.classList.toggle("active", b === btn)
      );
      document.querySelectorAll(".tabPanel").forEach((p) =>
        p.classList.toggle("active", p.id === id)
      );

      // --- IMPORTANT: canvas inside hidden tabs needs a resize after it becomes visible
      if (id === "tabXY") {
        requestAnimationFrame(() => {
          try {
            if (typeof resizeXY === "function") resizeXY();
            if (typeof clearXY === "function") clearXY();
          } catch (_) {
            // fail silently (keeps UI.js robust)
          }
        });
      }

      // --- IMPORTANT: PQ phasor canvas needs redraw after tab becomes visible
      if (id === "tabPQ") {
        requestAnimationFrame(() => {
          try {
            // updatePowerPanel() redraws the phasor plot (drawPhasorPlot is called inside)
            if (typeof updatePowerPanel === "function") updatePowerPanel();
          } catch (_) {
            // fail silently
          }
        });
      }
    });
  });

  // --- XY Presets ------------------------------------------------------------
  function applyXYPreset(kind){
    const xSel = document.querySelector("#xyX");
    const ySel = document.querySelector("#xyY");
    if(!xSel || !ySel) return;

    if(kind === "pf"){         // v vs i  (PF ellipse)
      xSel.value = "vSrc";
      ySel.value = "i";
    } else if(kind === "ivl"){ // i vs vL
      xSel.value = "i";
      ySel.value = "vL";
    } else if(kind === "vvl"){ // v vs vL
      xSel.value = "vSrc";
      ySel.value = "vL";
    } else if(kind === "swap"){ // swap current X/Y
      const tmp = xSel.value;
      xSel.value = ySel.value;
      ySel.value = tmp;
    }

    // if you already added these functions, this keeps the display snappy
    if(typeof resizeXY === "function") resizeXY();
    if(typeof clearXY === "function") clearXY();
  }

  // Bind buttons
  document.querySelectorAll(".xyPresetBtn").forEach((btn) => {
    btn.addEventListener("click", () => {
      applyXYPreset(btn.dataset.xypreset);
    });
  });

  // -------------------------
  // Source mode radios
  // -------------------------
  const rPulse = $("#srcPulse");
  const rSine  = $("#srcSine");

  // --- Startup sync: STATE -> DOM (deterministic) ---
  if (rPulse) rPulse.checked = (state.srcMode === "pulse");
  if (rSine)  rSine.checked  = (state.srcMode === "sine");

  applyModeUI();

  if (rPulse) rPulse.addEventListener("change", () => {
    if (rPulse.checked) {
      state.srcMode = "pulse";
      applyModeUI();
      state.refSnapshot = null;
    }
  });

  if (rSine) rSine.addEventListener("change", () => {
    if (rSine.checked) {
      state.srcMode = "sine";
      applyModeUI();
      state.refSnapshot = null;
    }
  });

  applyModeUI();

  // -------------------------
  // Set Ref / Clear Ref
  // -------------------------
  const btnSet = $("#setRef");
  const btnClr = $("#clearRef");

  if (btnSet) btnSet.addEventListener("click", () => {
    const tW = tau();

    const impulse = impulse_abs_vL_0_to(tW);
    const pkVL    = peak_abs_vL_0_to(tW);

    const { Psrc } = powerSplit();
    state.refSnapshot = { impulse, pkVL, Psrc, mode: state.srcMode };

    flashButton(btnSet, "Ref ✓");
  });

  const holdBtn = $("#holdBtn");

  if (holdBtn) {
    holdBtn.addEventListener("click", () => {
      state.hold = !state.hold;

      // Button label
      holdBtn.textContent = state.hold ? "RESUME" : "FREEZE";

      // Optional visual state class
      holdBtn.classList.toggle("isFrozen", state.hold);
    });

    // Initial sync on startup
    holdBtn.textContent = state.hold ? "RESUME" : "FREEZE";
    holdBtn.classList.toggle("isFrozen", state.hold);
  }

  function relocateScopeControlsForFocus(on){
    const scopeSection = document.getElementById("scopeSection");
    const focusDock    = document.getElementById("focusDock");
    const controlPanel = document.querySelector("aside.controlPanel");
    if(!scopeSection || !focusDock || !controlPanel) return;

    if(on){
      focusDock.appendChild(scopeSection);
    }else{
      // put it back near the bottom of the left panel
      controlPanel.appendChild(scopeSection);
    }
  }

  // -------------------------
  // Focus mode (Scope + Dock)
  // -------------------------
  const focusBtn = $("#focusBtn");

  // persist scroll position across clicks
  let preFocusScrollY = 0;

  function updateFocusVars(){
    const topbar = document.querySelector(".topbar");
    if (topbar) {
      const topH = Math.ceil(topbar.getBoundingClientRect().height);
      document.documentElement.style.setProperty("--topbarH", `${topH}px`);
    }

    const dockH = Math.max(160, Math.min(260, Math.round(window.innerHeight * 0.22)));
    document.documentElement.style.setProperty("--focusDockH", `${dockH}px`);
  }

  if (focusBtn) {
    focusBtn.addEventListener("click", () => {
      const body = document.body;

      const turningOn = !body.classList.contains("focusMode");
      if (turningOn) {
        preFocusScrollY = window.scrollY || 0;
      }

      body.classList.toggle("focusMode");

      const on = body.classList.contains("focusMode");
      focusBtn.textContent = on ? "Exit Focus" : "Focus";

      // relocate scope controls into dock
      relocateScopeControlsForFocus(on);

      // After layout reflow, compute true header/dock heights + force correct scroll
      requestAnimationFrame(() => {
        updateFocusVars();

        if (on) {
          // hard force: always show scope on top in focus mode
          document.documentElement.scrollTop = 0;
          document.body.scrollTop = 0;
          window.scrollTo(0, 0);

          const stage = document.querySelector(".stageTop");
          if (stage) stage.scrollIntoView({ block: "start", inline: "nearest" });
        } else {
          window.scrollTo(0, preFocusScrollY);
        }

        // Resize scope after layout change
        if (typeof resizeScope === "function") {
          resizeScope();
          setTimeout(() => resizeScope(), 120);
        }
      });
    });
    // keep focus layout stable on window resize
    window.addEventListener("resize", () => {
      if (document.body.classList.contains("focusMode")) {
        requestAnimationFrame(() => {
          updateFocusVars();
          if (typeof resizeScope === "function") resizeScope();
        });
      }
    });
  }

  // -------------------------
  // Scope background select (Video ↔ Dark)
  // -------------------------
  (function bindScopeBackground(){
    const sel = $("#bgMode");
    if(!sel) return;

    // restore
    const saved = localStorage.getItem("scopeBgMode"); // "video" | "dark"
    const mode = (saved === "dark") ? "dark" : "video";

    sel.value = mode;
    document.body.classList.toggle("bgDark", mode === "dark");

    sel.addEventListener("change", () => {
      const m = sel.value === "dark" ? "dark" : "video";
      document.body.classList.toggle("bgDark", m === "dark");
      localStorage.setItem("scopeBgMode", m);

      // optional: scope resize after background swap (usually not needed)
      // if (typeof resizeScope === "function") resizeScope();
    });
  })();

  if (btnClr) btnClr.addEventListener("click", () => {
    state.refSnapshot = null;
    flashButton(btnClr, "Cleared");
  });
}
