// app/time_domain.js
// Truth Accessor (time-domain abstraction)
//
// Purpose:
// - Provide a single, topology-aware, mode-aware sampler:
//     sampleAt(t) -> { vSrc, i, vL, vC }
// - Lives BELOW metrics (imports only state/core/physics kernels).
// - Allows wave_engine/rendering/measurement to consume "truth" uniformly.
//
// HARD RULES:
// - No numeric changes (reuse exact formulas from wave_engine.js)
// - No behavioral changes
// - No circular imports

import { state } from "./state.js";
import { tau, Rtotal, periodPulse, duty } from "./core/derived.js";

// Physics kernels
import { sineParams } from "./rl_sine.js";
import { rlcSineParamsSeries, rlcSineParamsParallelRL } from "./rlc_sine.js";

import {
  steadyI0,
  i_on,
  vL_on,
  i_off,
  vL_off
} from "./rl_pulse.js";

import {
  rlcPulseSteadyState,
  rlcPulseParallelSteadyState
} from "./rlc_pulse.js";

// Returns a topology+mode sampler.
// NOTE: Caller decides how to sample (arrays, cursor, integrators, etc).
export function makeTruthAccessor(){
  const tt = tau();

  // ---------- SINE ----------
  if(state.srcMode === "sine"){
    const C = Math.max(0, state.C || 0);
    const topology = state.topology || "seriesRLC";

    // Series RLC sine
    if(topology === "seriesRLC" && C > 0){
      const sp = rlcSineParamsSeries();

      return {
        hasVC: true,
        tau: tt,
        sampleAt(t){
          const vs = sp.Vpk * Math.sin(sp.w * t);
          const is = sp.Ipk * Math.sin(sp.w * t - sp.phi);

          // vL = L di/dt
          const vLk = sp.w * sp.L * sp.Ipk * Math.cos(sp.w * t - sp.phi);
          const vRk = is * sp.R;
          const vCk = vs - vRk - vLk; // guarantees KVL in the rendered signals

          return { vSrc: vs, i: is, vL: vLk, vC: vCk };
        }
      };
    }

    // Parallel (R+L)||C sine
    if(topology === "C_parallel_RL" && C > 0){
      const sp = rlcSineParamsParallelRL();

      return {
        hasVC: true,
        tau: tt,
        sampleAt(t){
          const vs = sp.Vpk * Math.sin(sp.w * t);

          // total source current (phasor via Zeq)
          const is = sp.Ipk * Math.sin(sp.w * t - sp.phi);

          // capacitor voltage equals source voltage in this topology
          const vCk = vs;

          // Inductor voltage inside RL branch:
          // i_RL(t) = Irl_pk sin(ωt - phiRL)
          // vL(t) = L di/dt = ωL Irl_pk cos(ωt - phiRL)
          const vLk = sp.w * sp.L * sp.Irl_pk * Math.cos(sp.w * t - sp.phiRL);

          return { vSrc: vs, i: is, vL: vLk, vC: vCk };
        }
      };
    }

    // Fallback: original RL sine
    {
      const sp = sineParams();
      const L = (sp.L != null) ? sp.L : state.L;

      return {
        hasVC: false,
        tau: tt,
        sampleAt(t){
          const vs = state.V * Math.sin(sp.w * t);
          const is = sp.Ipk * Math.sin(sp.w * t - sp.phi);
          const vLk = sp.w * L * sp.Ipk * Math.cos(sp.w * t - sp.phi);
          return { vSrc: vs, i: is, vL: vLk, vC: 0 };
        }
      };
    }
  }

  // ---------- PULSE ----------
  const C = Math.max(0, state.C || 0);
  const topology = state.topology || "seriesRLC";

  // Parallel (R+L)||C pulse
  if(topology === "C_parallel_RL" && C > 0){
    const sol = rlcPulseParallelSteadyState();
    const T = sol.T || periodPulse();
    const wrap = (t)=> ((t % T) + T) % T;

    return {
      hasVC: true,
      tau: tt,
      sampleAt(t){
        const tm = wrap(t);
        const vs = sol.vSrcAt(tm);
        const vc = sol.vCAt(tm);
        const is = sol.iAt(tm);
        const vl = sol.vLAt(tm);
        return { vSrc: vs, i: is, vL: vl, vC: vc };
      }
    };
  }

  // Series RLC pulse
  if(topology === "seriesRLC" && C > 0){
    const sol = rlcPulseSteadyState();
    const T = sol.T || periodPulse();
    const wrap = (t)=> ((t % T) + T) % T;

    return {
      hasVC: true,
      tau: tt,
      sampleAt(t){
        const tm = wrap(t);
        const vs = sol.vSrcAt(tm);
        const st = sol.stateAt(tm);

        const is = st[0];
        const vc = st[1];
        const vl = vs - sol.R*is - vc; // same as rlcPulseWave

        return { vSrc: vs, i: is, vL: vl, vC: vc };
      }
    };
  }

  // Existing RL pulse (no C)
  {
    const T   = periodPulse();
    const Ton = T * duty();
    const Rt  = Rtotal();

    const Iinf = state.V / Rt;
    const I0   = steadyI0();
    const Iend = i_on(Ton, I0, Iinf, tt);
    const wrap = (t)=> ((t % T) + T) % T;

    return {
      hasVC: false,
      tau: tt,
      sampleAt(t){
        const tm = wrap(t);

        const vs = (tm < Ton) ? state.V : 0;
        let is, vLk;

        if(tm < Ton){
          is  = i_on(tm, I0, Iinf, tt);
          vLk = vL_on(tm, I0, Iinf, tt);
        }else{
          const t2 = tm - Ton;
          is  = i_off(t2, Iend, tt);
          vLk = vL_off(t2, Iend, tt);
        }

        return { vSrc: vs, i: is, vL: vLk, vC: 0 };
      }
    };
  }
}