// tests/physics_smoke.js
// Browser-run, DOM-free physics sanity checks for RLC Analyzer.
// Usage: import { runAllTests } and run in a minimal test page (see index.html :contentReference[oaicite:1]{index=1}).

import { state } from "../state.js";
import { Rtotal, tau, periodPulse, duty } from "../core/derived.js";
import { sineParams } from "../rl_sine.js";
import { rlcSineParamsParallelRL } from "../rlc_sine.js";
import { steadyI0 } from "../rl_pulse.js";
import { buildWaves } from "../wave_engine.js";

function isFiniteNumber(x){
  return typeof x === "number" && Number.isFinite(x);
}

function approx(a, b, rel = 1e-6, abs = 1e-12){
  const da = Math.abs(a - b);
  const m = Math.max(abs, rel * Math.max(1, Math.abs(a), Math.abs(b)));
  return da <= m;
}

function assert(cond, msg){
  if(!cond) throw new Error(msg);
}

function assertFinite(x, msg){
  assert(isFiniteNumber(x), `${msg} (got ${x})`);
}

function snapshotState(){
  try { return structuredClone(state); } catch(_){ return JSON.parse(JSON.stringify(state)); }
}

function restoreState(snap){
  for(const k of Object.keys(state)) delete state[k];
  for(const k of Object.keys(snap)) state[k] = snap[k];
}

function setCommonDefaults(){
  // pick stable defaults (won't depend on UI)
  state.L = 0.027;
  state.Rcoil = 3.25;
  state.Rload = 16.50;
  state.V = 470;

  state.f = 200;
  state.dutyPct = 5.0;

  state.fs = 200;

  state.timeDiv_ms = 1.0;
  state.srcMode = "sine";
  state.hold = false;

  state.C = 0;
  state.topology = "seriesRLC";
}

function testDerivedHelpers(){
  setCommonDefaults();

  const Rt = Rtotal();
  assertFinite(Rt, "Rtotal finite");
  assert(Rt > 0, "Rtotal > 0");

  const tt = tau();
  assertFinite(tt, "tau finite");
  assert(tt > 0, "tau > 0");

  // increasing R increases Rtotal and reduces tau
  const tau0 = tt;
  state.Rload *= 2;
  assert(Rtotal() > Rt, "Rtotal increases when Rload increases");
  assert(tau() < tau0, "tau decreases when Rtotal increases");
}

function testRLSineSanity(){
  setCommonDefaults();

  // L -> 0 => pure R
  state.L = 0;
  const p0 = sineParams();
  assertFinite(p0.Z, "Z finite (L=0)");
  assert(approx(p0.Z, Rtotal(), 1e-9, 1e-12), "Z ≈ Rtotal when L=0");
  assert(Math.abs(p0.phi) < 1e-9, "phi ≈ 0 when L=0");
  assertFinite(p0.Irms, "Irms finite (L=0)");

  // higher frequency => higher XL => lower current (for inductive RL)
  setCommonDefaults();
  state.L = 0.027;
  state.fs = 50;
  const p50 = sineParams();
  state.fs = 5000;
  const p5k = sineParams();
  assert(p5k.XL > p50.XL, "XL increases with frequency");
  assert(p5k.Irms < p50.Irms, "Irms decreases with frequency (inductive RL)");
  assert(p5k.phi > p50.phi, "phase lag increases with frequency (inductive RL)");
}

function testRLPulseSteady(){
  setCommonDefaults();

  // duty -> 0 => I0 -> ~0
  state.dutyPct = 0;
  const i0a = steadyI0();
  assertFinite(i0a, "steadyI0 finite at duty=0");
  assert(Math.abs(i0a) < 1e-9, "steadyI0 ≈ 0 at duty=0");

  // larger R => smaller I0 (for same V)
  setCommonDefaults();
  state.dutyPct = 10;
  const i0_1 = steadyI0();
  state.Rload *= 2;
  const i0_2 = steadyI0();
  assert(i0_2 < i0_1, "steadyI0 decreases when R increases");
}

function testWaveEngineFrame(){
  setCommonDefaults();

  const tStart = 0;
  const span = 0.01;
  const N = 1200;

  // sine mode
  state.srcMode = "sine";
  const a = buildWaves({ tStart, span, N });
  assert(a && a.i && a.vSrc, "buildWaves returns arrays (sine)");
  assert(a.i.length === N, "i length matches N (sine)");
  assert(a.vSrc.length === N, "vSrc length matches N (sine)");

  // pulse mode
  state.srcMode = "pulse";
  const b = buildWaves({ tStart, span, N });
  assert(b && b.i && b.vSrc, "buildWaves returns arrays (pulse)");
  assert(b.i.length === N, "i length matches N (pulse)");
  assert(b.vSrc.length === N, "vSrc length matches N (pulse)");

  // no NaNs (pulse)
  for(const name of ["i","vSrc","vL","vC"]){
    const arr = b[name];
    if(!arr) continue;
    for(let k=0;k<arr.length;k++){
      const x = arr[k];
      if(!Number.isFinite(x)) throw new Error(`${name}[${k}] not finite (pulse): ${x}`);
    }
  }
}

function testPowerTriangleIdentity_RL_Sine(){
  // Checks: S^2 ≈ P^2 + Q^2 for sine RL using phasor formulas.
  setCommonDefaults();
  state.srcMode = "sine";
  state.C = 0;

  const p = sineParams();
  const Vrms = p.Vrms;
  const Irms = p.Irms;

  const S = Vrms * Irms;
  const P = Vrms * Irms * Math.cos(p.phi);
  const Q = Vrms * Irms * Math.sin(p.phi);

  assertFinite(S, "S finite");
  assertFinite(P, "P finite");
  assertFinite(Q, "Q finite");

  assert(approx(S*S, P*P + Q*Q, 2e-9, 1e-12), "Power triangle: S^2 ≈ P^2 + Q^2");
}

function testEnergyBalance_RL_Sine_TimeDomain(){
  // Over whole periods (steady-state sine RL): Esrc ≈ ER and ΔWL ≈ 0
  setCommonDefaults();
  state.srcMode = "sine";
  state.C = 0;

  const fs = Math.max(1e-12, state.fs);
  const T = 1 / fs;
  const periods = 20;
  const span = periods * T;
  const tStart = 0;
  const N = 8000;

  const w = buildWaves({ tStart, span, N });
  assert(w && w.i && w.vSrc, "buildWaves returns arrays (energy balance RL)");

  const dt = w.dt;
  assertFinite(dt, "dt finite (energy balance RL)");
  assert(dt > 0, "dt > 0 (energy balance RL)");

  const i = w.i;
  const v = w.vSrc;
  const R = Rtotal();

  let Esrc = 0;
  let ER   = 0;

  for(let k=0;k<N;k++){
    const ik = i[k];
    const vk = v[k];
    Esrc += (vk * ik) * dt;
    ER   += (ik * ik * R) * dt;
  }

  assertFinite(Esrc, "Esrc finite (RL)");
  assertFinite(ER, "ER finite (RL)");

  const scale = Math.max(1e-9, Math.abs(Esrc), Math.abs(ER));
  const relErr = Math.abs(Esrc - ER) / scale;
  assert(relErr < 2e-3, `Energy balance RL: Esrc ≈ ER (relErr=${relErr})`);

  const WL0 = 0.5 * state.L * (i[0]*i[0]);
  const WL1 = 0.5 * state.L * (i[N-1]*i[N-1]);
  const dWL = WL1 - WL0;

  const dRel = Math.abs(dWL) / Math.max(1e-9, Math.abs(Esrc), Math.abs(ER));
  assert(dRel < 2e-3, `ΔW_L (RL) over whole periods ≈ 0 (rel=${dRel})`);
}

// ---------- Series RLC near resonance (sine) ----------
function setSeriesRLCNearResonance(){
  setCommonDefaults();
  state.srcMode = "sine";
  state.topology = "seriesRLC";

  // Keep f0 in the hundreds of Hz (fast + stable numerics)
  state.L = 0.027;
  state.C = 4.7e-6;

  const L = Math.max(1e-18, state.L);
  const C = Math.max(1e-18, state.C);

  const f0 = 1 / (2 * Math.PI * Math.sqrt(L * C));

  // slightly detuned to stress sign conventions
  state.fs = f0 * 1.02;

  return { f0, fs: state.fs };
}

function testPowerTriangleIdentity_RLC_Sine_Series_NearResonance(){
  const { f0, fs } = setSeriesRLCNearResonance();

  const R = Rtotal();
  const w = 2 * Math.PI * Math.max(1e-12, state.fs);
  const XL = w * state.L;
  const Xc = 1 / (w * Math.max(1e-18, state.C));
  const X  = XL - Xc;

  const Z = Math.sqrt(R*R + X*X);
  const phi = Math.atan2(X, R);

  const Vpk = state.V;
  const Vrms = Vpk / Math.SQRT2;
  const Ipk = Vpk / Math.max(1e-12, Z);
  const Irms = Ipk / Math.SQRT2;

  const S = Vrms * Irms;
  const P = Vrms * Irms * Math.cos(phi);
  const Q = Vrms * Irms * Math.sin(phi);

  assertFinite(f0, "f0 finite (RLC)");
  assertFinite(fs, "fs finite (RLC)");
  assertFinite(Z, "Z finite (RLC)");
  assertFinite(phi, "phi finite (RLC)");

  assert(approx(S*S, P*P + Q*Q, 2e-9, 1e-12), "Power triangle (series RLC): S^2 ≈ P^2 + Q^2");
}

function testEnergyBalance_RLC_Sine_Series_NearResonance_TimeDomain(){
  // Over whole periods, for steady-state series RLC:
  // Esrc ≈ ER and Δ(WL + WC) ≈ 0
  setSeriesRLCNearResonance();

  const fs = Math.max(1e-12, state.fs);
  const T = 1 / fs;

  const periods = 30;
  const span = periods * T;
  const tStart = 0;
  const N = 12000;

  const w = buildWaves({ tStart, span, N });
  assert(w && w.i && w.vSrc, "buildWaves returns arrays (energy balance series RLC)");
  assert(w.vC, "buildWaves provides vC array (series RLC)");

  const dt = w.dt;
  assertFinite(dt, "dt finite (energy balance series RLC)");
  assert(dt > 0, "dt > 0 (energy balance series RLC)");

  const i  = w.i;
  const vS = w.vSrc;
  const vC = w.vC;

  const R = Rtotal();

  let Esrc = 0;
  let ER   = 0;

  for(let k=0;k<N;k++){
    const ik = i[k];
    const vk = vS[k];
    Esrc += (vk * ik) * dt;
    ER   += (ik * ik * R) * dt;

    if(!Number.isFinite(ik) || !Number.isFinite(vk) || !Number.isFinite(vC[k])){
      throw new Error(`Non-finite waveform sample at k=${k}`);
    }
  }

  assertFinite(Esrc, "Esrc finite (series RLC)");
  assertFinite(ER, "ER finite (series RLC)");

  const scale = Math.max(1e-9, Math.abs(Esrc), Math.abs(ER));
  const relErr = Math.abs(Esrc - ER) / scale;
  assert(relErr < 5e-3, `Energy balance series RLC: Esrc ≈ ER (relErr=${relErr})`);

  const WL0 = 0.5 * state.L * (i[0]*i[0]);
  const WL1 = 0.5 * state.L * (i[N-1]*i[N-1]);

  const WC0 = 0.5 * state.C * (vC[0]*vC[0]);
  const WC1 = 0.5 * state.C * (vC[N-1]*vC[N-1]);

  const dW = (WL1 + WC1) - (WL0 + WC0);
  const dRel = Math.abs(dW) / Math.max(1e-9, Math.abs(Esrc), Math.abs(ER));
  assert(dRel < 5e-3, `Δ(WL+WC) series RLC over whole periods ≈ 0 (rel=${dRel})`);
}

// ---------- Parallel topology (R+L)||C (sine) ----------
function setParallelRLCAtResonance(){
  setCommonDefaults();
  state.srcMode = "sine";
  state.topology = "C_parallel_RL";

  // keep it in safe region: w0^2 = 1/(LC) - (R/L)^2  must be > 0
  state.L = 0.027;
  state.C = 4.7e-6;

  const R = Rtotal();
  const L = Math.max(1e-18, state.L);
  const C = Math.max(1e-18, state.C);

  const w0sq = (1 / (L*C)) - (R*R)/(L*L);
  assert(w0sq > 0, `parallel resonance exists (w0^2>0), got w0^2=${w0sq}`);

  const w0 = Math.sqrt(w0sq);
  const f0 = w0 / (2*Math.PI);
  state.fs = f0;

  return { f0 };
}

function testParallelRLC_Sine_Resonance_PhaseAndImpedancePeak(){
  const { f0 } = setParallelRLCAtResonance();

  const sp0 = rlcSineParamsParallelRL();
  assertFinite(sp0.Zmag, "Zmag finite (parallel @ f0)");
  assertFinite(sp0.phi, "phi finite (parallel @ f0)");

  // At parallel resonance the net susceptance is ~0 => phi ~ 0 (mostly resistive)
  assert(Math.abs(sp0.phi) < (3 * Math.PI/180), `phi near 0 at parallel resonance (phi=${(sp0.phi*180/Math.PI).toFixed(3)}°)`);

  // Impedance peaks around resonance (compare against detuned points)
  state.fs = f0 * 0.6;
  const spLo = rlcSineParamsParallelRL();
  state.fs = f0 * 1.6;
  const spHi = rlcSineParamsParallelRL();

  assert(sp0.Zmag > spLo.Zmag, "Zmag peaks vs low detune (parallel)");
  assert(sp0.Zmag > spHi.Zmag, "Zmag peaks vs high detune (parallel)");
}

// ---------- Pulse RLC (series + parallel) ----------
function testPulseRLC_Series_PeriodicityAndEnergyBalance(){
  setCommonDefaults();
  state.srcMode = "pulse";
  state.topology = "seriesRLC";

  // enable RLC pulse solver
  state.L = 0.027;
  state.C = 4.7e-6;

  const T = periodPulse();
  assertFinite(T, "T finite (pulse series RLC)");
  assert(T > 0, "T > 0 (pulse series RLC)");

  const N = 12000;
  const w = buildWaves({ tStart: 0, span: T, N });

  assert(w && w.i && w.vSrc && w.vC, "buildWaves returns i/vSrc/vC (pulse series RLC)");
  assert(w.i.length === N && w.vSrc.length === N && w.vC.length === N, "array lengths OK (pulse series RLC)");

  const i0 = w.i[0], i1 = w.i[N-1];
  const vC0 = w.vC[0], vC1 = w.vC[N-1];

  // steady-periodic should match at ends (t=0 vs t=T)
  const iScale = Math.max(1e-9, Math.abs(i0), Math.abs(i1));
  const vScale = Math.max(1e-6, Math.abs(vC0), Math.abs(vC1));

  assert(Math.abs(i1 - i0)/iScale < 5e-3, `periodic i(T)≈i(0) (rel=${Math.abs(i1-i0)/iScale})`);
  assert(Math.abs(vC1 - vC0)/vScale < 5e-3, `periodic vC(T)≈vC(0) (rel=${Math.abs(vC1-vC0)/vScale})`);

  // Energy balance over one period:
  // ∫ v*i dt ≈ ∫ i^2*Rtotal dt  (since ΔWL≈0 and ΔWC≈0 in steady periodic)
  const dt = w.dt;
  const R = Rtotal();

  let Esrc = 0, ER = 0;
  for(let k=0;k<N;k++){
    const ik = w.i[k];
    const vk = w.vSrc[k];
    Esrc += vk*ik*dt;
    ER   += ik*ik*R*dt;
    if(!Number.isFinite(ik) || !Number.isFinite(vk) || !Number.isFinite(w.vC[k])){
      throw new Error(`Non-finite sample at k=${k} (pulse series RLC)`);
    }
  }

  const scale = Math.max(1e-6, Math.abs(Esrc), Math.abs(ER));
  const relErr = Math.abs(Esrc - ER)/scale;
  assert(relErr < 8e-3, `pulse series RLC energy: Esrc≈ER (relErr=${relErr})`);
}

function testPulseRLC_Parallel_VCEqualsVS_AndFinite(){
  setCommonDefaults();
  state.srcMode = "pulse";
  state.topology = "C_parallel_RL";

  state.L = 0.027;
  state.C = 4.7e-6;

  const T = periodPulse();
  assertFinite(T, "T finite (pulse parallel)");
  assert(T > 0, "T > 0 (pulse parallel)");

  const N = 8000;
  const w = buildWaves({ tStart: 0, span: T, N });
  assert(w && w.vC && w.vSrc && w.i, "buildWaves returns vC/vSrc/i (pulse parallel)");
  assert(w.vC.length === N && w.vSrc.length === N, "array lengths OK (pulse parallel)");

  // vC must equal vSrc in this topology (render convention)
  let maxDiff = 0;
  for(let k=0;k<N;k++){
    const d = Math.abs(w.vC[k] - w.vSrc[k]);
    if(d > maxDiff) maxDiff = d;

    // also ensure no NaN/Inf
    for(const name of ["i","vSrc","vL","vC"]){
      const x = w[name]?.[k];
      if(x == null) continue;
      if(!Number.isFinite(x)) throw new Error(`${name}[${k}] not finite (pulse parallel): ${x}`);
    }
  }
  const vScale = Math.max(1e-6, Math.max(...w.vSrc.map(x=>Math.abs(x))));
  assert((maxDiff / vScale) < 1e-9, `parallel pulse: vC==vSrc (maxRel=${maxDiff/vScale})`);
}

export function runAllTests(){
  const snap = snapshotState();
  const results = [];
  const start = performance.now();

  function run(name, fn){
    const t0 = performance.now();
    try{
      fn();
      results.push({ name, ok: true, ms: performance.now() - t0 });
    }catch(err){
      results.push({ name, ok: false, ms: performance.now() - t0, err: String(err?.message || err) });
    }
  }

  try{
    run("Derived helpers: Rtotal/tau", testDerivedHelpers);
    run("RL Sine sanity", testRLSineSanity);
    run("RL Pulse steady-state", testRLPulseSteady);
    run("Wave engine frame build", testWaveEngineFrame);

    run("Power triangle identity (RL sine)", testPowerTriangleIdentity_RL_Sine);
    run("Energy balance (RL sine, time-domain)", testEnergyBalance_RL_Sine_TimeDomain);

    run("Power triangle identity (series RLC near resonance)", testPowerTriangleIdentity_RLC_Sine_Series_NearResonance);
    run("Energy balance (series RLC near resonance)", testEnergyBalance_RLC_Sine_Series_NearResonance_TimeDomain);

    run("Parallel RLC sine resonance: phase + Z peak", testParallelRLC_Sine_Resonance_PhaseAndImpedancePeak);

    run("Pulse series RLC: periodicity + energy balance", testPulseRLC_Series_PeriodicityAndEnergyBalance);
    run("Pulse parallel (R+L)||C: vC==vSrc + finite", testPulseRLC_Parallel_VCEqualsVS_AndFinite);
  }finally{
    restoreState(snap);
  }

  const totalMs = performance.now() - start;
  const pass = results.filter(r => r.ok).length;
  const fail = results.length - pass;

  return { pass, fail, totalMs, results };
}