// scope.js
import { $, clamp, fmtDivMs, fmt } from "./dom.js";
import { state } from "./state.js";
import { tau, Rtotal, periodPulse, periodSine, duty } from "./core/derived.js";
import { buildWaves } from "./wave_engine.js";
import { impulse_abs_vL_0_to, peak_abs_vL_0_to, windowMagneticMetrics } from "./metrics.js";
import {
  updateMeter,
  updatePkVLmeter,
  updatePowerPanel,
  updateDerived,
  updatePeriodicityBadge
} from "./panels.js";

let scope = null;
let ctx = null;

// last rendered scope data for cursor readout
let lastScopeData = null;

// autoscale smoothing factor (scope vertical stability)
const AUTOSCALE_ALPHA = 0.10;

// smoothed autoscale (prevents flicker)
let Vfs_s = null;
let iPeak_s = null;
let vLPeak_s = null;
let vCPeak_s = null;

// cursor state (for dashed line)
let cursorX = null;       // CSS px inside canvas
let cursorActive = false;

// cursor sample (computed in ui.js, reused for LAB overlays)
let cursorSample = null; // { active, vsrc, i, vL, vR, vC? }
export function setCursorSample(s){ cursorSample = s || null; }
export function clearCursorSample(){ cursorSample = null; }

// running window timebase (scope "RUN/STOP")
let runT_s = 0;          // current left-edge time of the displayed window (seconds)
let lastNow_s = null;    // for dt integration

export function initScope(canvasEl){
  scope = canvasEl;
  if(!scope) throw new Error("initScope: canvas element missing");
  ctx = scope.getContext('2d', { alpha:true, desynchronized:true });
  resizeScope();
}

export function resizeScope(){
  if(!scope) return;
  const dpr = Math.min(2, window.devicePixelRatio || 1);
  const rect = scope.getBoundingClientRect();
  const w = Math.max(1, Math.floor(rect.width * dpr));
  const h = Math.max(1, Math.floor(rect.height * dpr));
  scope.width = w;
  scope.height = h;
  scope.dataset.dpr = String(dpr);
}

export function getLastScopeData(){ return lastScopeData; }

export function setCursorState(xCssPx, active){
  cursorX = xCssPx;
  cursorActive = !!active;
}

export function clearCursorState(){
  cursorX = null;
  cursorActive = false;
}

function roundRect(x,y,w,h,r,fill,stroke){
  ctx.beginPath();
  const rr = Math.min(r, w/2, h/2);
  ctx.moveTo(x+rr, y);
  ctx.arcTo(x+w, y, x+w, y+h, rr);
  ctx.arcTo(x+w, y+h, x, y+h, rr);
  ctx.arcTo(x, y+h, x, y, rr);
  ctx.arcTo(x, y, x+w, y, rr);
  ctx.closePath();
  if(fill) ctx.fill();
  if(stroke) ctx.stroke();
}

function xFromU(u, x0, w){ return x0 + u*w; }

function fmtTimeAxis(t){
  const at = Math.abs(t);
  if (at < 1e-6) return `${(t*1e9).toFixed(0)}ns`;
  if (at < 1e-3) return `${(t*1e6).toFixed(1)}µs`;
  if (at < 1)    return `${(t*1e3).toFixed(2)}ms`;
  return `${t.toFixed(3)}s`;
}

function drawYAxisLabels({ x0, y0, w, h, divY }, perDiv, unit, color, offsetX = 0){
  if(!state.hold) return;
  ctx.save();
  ctx.font = '900 11px ui-sans-serif, system-ui';
  ctx.fillStyle = color;
  ctx.textAlign = 'right';       // important
  ctx.textBaseline = 'middle';

  const baseX = x0 + w - 8 - offsetX;  // right side

  for(let j=0;j<=divY;j++){
    const y = y0 + (j/divY)*h;
    const val = (divY/2 - j) * perDiv;
    ctx.fillText(`${val.toFixed(2)} ${unit}`, baseX, y);
  }

  ctx.restore();
}

function drawXAxisLabels({ x0, y0, w, h, divX }, tStart, span){
  if(!state.hold) return;
  ctx.save();
  ctx.font = '900 11px ui-sans-serif, system-ui';
  ctx.fillStyle = 'rgba(151,166,178,.85)';
  ctx.textAlign = 'center';
  ctx.textBaseline = 'top';

  const y = y0 + h + 8;
  for(let i=0;i<=divX;i++){
    const x = x0 + (i/divX)*w;
    const t = tStart + (i/divX)*span;
    ctx.fillText(fmtTimeAxis(t), x, y);
  }
  ctx.restore();
}

function drawTrace(samples, x0, yMid, scale, stroke, lw, w){
  ctx.save();
  ctx.strokeStyle = stroke;
  ctx.lineWidth = lw;
  ctx.beginPath();
  for(let i=0;i<samples.length;i++){
    const u = i/(samples.length-1);
    const x = xFromU(u, x0, w);
    const y = yMid - samples[i]*scale;
    if(i===0) ctx.moveTo(x,y); else ctx.lineTo(x,y);
  }
  ctx.stroke();
  ctx.restore();
}

function drawHLine(x0, x1, y, color){
  ctx.save();
  ctx.lineWidth = 1;
  ctx.setLineDash([5,5]);
  ctx.strokeStyle = color;
  ctx.beginPath();
  ctx.moveTo(x0, y);
  ctx.lineTo(x1, y);
  ctx.stroke();
  ctx.restore();
}

function drawTag(xRight, y, text, color, yTop, yBot){
  ctx.save();
  ctx.font = '900 11px ui-sans-serif, system-ui';
  const padX = 8;
  const tw = ctx.measureText(text).width;
  const w = tw + padX*2;
  const h = 20;

  const x = xRight - w - 10;
  const yClamped = clamp(y - h/2, yTop + 6, yBot - h - 6);

  ctx.fillStyle = 'rgba(0,0,0,.55)';
  ctx.strokeStyle = color;
  roundRect(x, yClamped, w, h, 7, true, true);

  ctx.fillStyle = color;
  ctx.fillText(text, x + padX, yClamped + 14);
  ctx.restore();
}


// --- robustness helpers (prevents NaN autoscale killing traces) ---
function maxAbsFinite(arr, fallback){
  let m = 0;
  for(let i=0;i<arr.length;i++){
    const v = arr[i];
    if(!Number.isFinite(v)) continue;
    const a = Math.abs(v);
    if(a > m) m = a;
  }
  return (m > 0) ? m : (fallback ?? 0);
}

function sanitizeInPlace(arr){
  // replace non-finite samples with last finite value (keeps trace continuous)
  let last = 0;
  for(let i=0;i<arr.length;i++){
    const v = arr[i];
    if(Number.isFinite(v)) last = v;
    else arr[i] = last;
  }
}

export function renderScope(){
  if(!scope || !ctx) return;

  const perf = window.__perf || null;

  // ==============
  // RENDER (setup / clear)
  // ==============
  if(perf?.tic) perf.tic("render");

  const dpr = parseFloat(scope.dataset.dpr || '1');
  const W = scope.width / dpr;
  const H = scope.height / dpr;
  ctx.setTransform(dpr,0,0,dpr,0,0);
  ctx.clearRect(0,0,W,H);

  const pad = 22;
  const x0 = pad, y0 = pad+18, w = W - 2*pad, h = H - 2*pad - 22;
  const divX=10, divY=8;

  ctx.save();
  ctx.strokeStyle = 'rgba(30,42,58,.9)';
  ctx.lineWidth = 1;
  roundRect(x0, y0, w, h, 14, false, true);
  ctx.restore();

  ctx.save();
  for(let i=1;i<divX;i++){
    const x = x0 + (i/divX)*w;
    ctx.strokeStyle = (i===5) ? 'rgba(151,166,178,.28)' : 'rgba(30,42,58,.55)';
    ctx.beginPath(); ctx.moveTo(x,y0); ctx.lineTo(x,y0+h); ctx.stroke();
  }
  for(let j=1;j<divY;j++){
    const y = y0 + (j/divY)*h;
    ctx.strokeStyle = (j===4) ? 'rgba(151,166,178,.28)' : 'rgba(30,42,58,.55)';
    ctx.beginPath(); ctx.moveTo(x0,y); ctx.lineTo(x0+w,y); ctx.stroke();
  }
  ctx.restore();

  if(perf?.toc) perf.toc();

  const span = (state.timeDiv_ms*1e-3) * 10;
  const tt = tau();

  const N = Math.max(800, Math.min(4000, Math.floor(w)));
  const dt = span/(N-1);

  // ==============
  // PHYSICS (timebase + buildWaves + sanitize + autoscale targets)
  // ==============
  if(perf?.tic) perf.tic("physics");

  // --- running window start time (stable RUN/STOP) ---
  const now_s = performance.now() * 1e-3;
  if (lastNow_s === null) lastNow_s = now_s;

  if (!state.hold) {
    const dtRun = Math.min(0.05, Math.max(0, now_s - lastNow_s));
    runT_s += dtRun; // 1.0x real-time feel
  }
  lastNow_s = now_s;

  // keep a little pre-trigger look by shifting window left
  const tStart = runT_s - 0.2 * span;

  // --- build waves via wave_engine ---
  const built = buildWaves({ tStart, span, N });

  // local arrays (keep original behavior)
  const vS  = built.vSrc.slice();
  const iS  = built.i.slice();
  const vLS = built.vL.slice();
  const vCS = built.vC ? built.vC.slice() : new Array(N).fill(0);

  // sanitize to keep drawing + autoscale stable if any NaNs appear
  sanitizeInPlace(vS);
  sanitizeInPlace(iS);
  sanitizeInPlace(vLS);
  sanitizeInPlace(vCS);

  // --- autoscale targets (instantaneous, NaN-safe) ---
  const Vfs_t    = Math.max(1,    maxAbsFinite(vS, 1)  * 1.15);
  const iPeak_t  = Math.max(0.01, maxAbsFinite(iS, 0.01));
  const vLPeak_t = Math.max(1,    maxAbsFinite(vLS, 1));
  const vCPeak_t = Math.max(1,    maxAbsFinite(vCS, 1));

  // --- smooth them (EMA) to avoid flicker ---
  const alpha = AUTOSCALE_ALPHA;

  if (Vfs_s === null)    Vfs_s = Vfs_t;
  if (iPeak_s === null)  iPeak_s = iPeak_t;
  if (vLPeak_s === null) vLPeak_s = vLPeak_t;
  if (vCPeak_s === null) vCPeak_s = vCPeak_t;

  Vfs_s    = Vfs_s    + alpha * (Vfs_t    - Vfs_s);
  iPeak_s  = iPeak_s  + alpha * (iPeak_t  - iPeak_s);
  vLPeak_s = vLPeak_s + alpha * (vLPeak_t - vLPeak_s);
  vCPeak_s = vCPeak_s + alpha * (vCPeak_t - vCPeak_s);

  // --- use smoothed values for scaling ---
  const Vfs = Vfs_s;
  const iPeak = iPeak_s;
  const vLPeak = vLPeak_s;
  const vCPeak = vCPeak_s;

  const VperDiv  = Vfs/4;
  const Vscale   = h/(divY*(Vfs/4));

  const IperDiv  = Math.max(1e-6, iPeak/3);
  const Iscale   = h/(divY*IperDiv);

  const VLperDiv = Math.max(1e-6, vLPeak/3);
  const VLscale  = h/(divY*VLperDiv);

  const VCperDiv = Math.max(1e-6, vCPeak/3);
  const VCscale  = h/(divY*VCperDiv);

  // ===== PERF: finish PHYSICS bucket here (buildWaves + sanitize + autoscale) =====
  if(perf?.toc) perf.toc();          // closes perf.tic("physics") from earlier

  // ===== PERF: start RENDER bucket (all canvas + DOM + panels) =====
  if(perf?.tic) perf.tic("render");

  const frame = {
    time: {
      tStart,
      span,
      dt,
      tau: tt
    },
    geom: {
      x0, y0, w, h, yMid
    },
    waves: {
      vSrc: vS,
      i: iS,
      vL: vLS,
      vC: vCS
    },
    scale: {
      Vscale,
      Iscale,
      VLscale,
      VCscale
    }
  };

  lastScopeData = frame;

  // violet window shading
  ctx.save();
  const win = Math.min(span, tt);
  ctx.fillStyle = 'rgba(178,123,255,.07)';
  const tEnd = tStart + span;

  if(state.srcMode === 'sine'){
    const T = periodSine();
    const n0 = Math.floor(tStart / T) - 1;
    const n1 = Math.ceil(tEnd / T) + 1;
    for(let n=n0;n<=n1;n++){
      const te = n*T;
      const a = (te - tStart)/span;
      const b = (te + win - tStart)/span;
      if(b < 0 || a > 1) continue;
      const xa = x0 + Math.max(0, a)*w;
      const xb = x0 + Math.min(1, b)*w;
      ctx.fillRect(xa, y0, xb-xa, h);
    }
  }else{
    const T = periodPulse();
    const n0 = Math.floor(tStart / T) - 1;
    const n1 = Math.ceil(tEnd / T) + 1;
    for(let n=n0;n<=n1;n++){
      const te = n*T;
      const a = (te - tStart)/span;
      const b = (te + win - tStart)/span;
      if(b < 0 || a > 1) continue;
      const xa = x0 + Math.max(0, a)*w;
      const xb = x0 + Math.min(1, b)*w;
      ctx.fillRect(xa, y0, xb-xa, h);
    }
  }
  ctx.restore();

  if($('#chV')?.checked)  drawTrace(vS,  x0, yMid, Vscale,  'rgba(77,227,255,0.95)',   2.2, w);
  if($('#chI')?.checked)  drawTrace(iS,  x0, yMid, Iscale,  'rgba(255,210,77,0.92)',   2.0, w);
  if($('#chVL')?.checked) drawTrace(vLS, x0, yMid, VLscale, 'rgba(178,123,255,0.86)',  2.0, w);
  if($('#chVC')?.checked) drawTrace(vCS, x0, yMid, VCscale, 'rgba(120,255,160,0.86)',  2.0, w);

  // --- Option B: axis tick labels ONLY in FROZEN ---
  if(state.hold){
    drawXAxisLabels({ x0, y0, w, h, divX }, tStart, span);

    let axisOffset = 0;
    if($('#chV')?.checked){
      drawYAxisLabels({ x0, y0, w, h, divY }, VperDiv, 'V', 'rgba(77,227,255,0.90)', axisOffset);
      axisOffset += 85;
    }
    if($('#chI')?.checked){
      drawYAxisLabels({ x0, y0, w, h, divY }, IperDiv, 'A', 'rgba(255,210,77,0.90)', axisOffset);
      axisOffset += 85;
    }
    if($('#chVL')?.checked){
      drawYAxisLabels({ x0, y0, w, h, divY }, VLperDiv, 'V', 'rgba(178,123,255,0.86)', axisOffset);
      axisOffset += 85;
    }
    if($('#chVC')?.checked){
      drawYAxisLabels({ x0, y0, w, h, divY }, VCperDiv, 'V', 'rgba(120,255,160,0.86)', axisOffset);
      axisOffset += 85;
    }
  }

  const modeTxt = (state.srcMode === 'sine')
    ? `Sine: f=${state.fs.toFixed(0)} Hz`
    : `Pulse: f=${state.f.toFixed(0)} Hz · Duty=${state.dutyPct.toFixed(2)}%`;

  const status = state.hold ? "FROZEN" : "LIVE";
  const runDot = state.hold ? "■" : (Math.floor(now_s * 2) % 2 ? "●" : " ");

  const chBits = [];
  if($('#chV')?.checked)  chBits.push(`CH1 ${VperDiv.toFixed(2)} V/div`);
  if($('#chI')?.checked)  chBits.push(`CH2 ${IperDiv.toFixed(3)} A/div`);
  if($('#chVL')?.checked) chBits.push(`CH3 ${VLperDiv.toFixed(2)} V/div`);
  if($('#chVC')?.checked) chBits.push(`CH4 ${VCperDiv.toFixed(2)} V/div`);

  // show channel scale info only in HOLD (as you requested)
  const chTxt = (state.hold && chBits.length) ? `  |  ${chBits.join('  ·  ')}` : "";

  const hudStatus = document.getElementById('hudStatus');
  if (hudStatus) {
    hudStatus.textContent =
      `${runDot} ${status}  |  ${modeTxt}  |  Time: ${fmtDivMs(state.timeDiv_ms)}${chTxt}  |  τ = ${(tt*1e3).toFixed(3)} ms`;
  }

  // panels + KPIs
  const nowImpulse = impulse_abs_vL_0_to(tt);
  updateMeter(nowImpulse);

  const nowPkVL = peak_abs_vL_0_to(tt);
  updatePkVLmeter(nowPkVL);

  updatePowerPanel();
  updateDerived();
  updatePeriodicityBadge();

  // badges
  ctx.save();
  ctx.font = '700 12px ui-sans-serif, system-ui';

  const badge1 = `∫|vL|dt (0→τ): ${nowImpulse.toExponential(2)} V·s`;
  const wmBadge = windowMagneticMetrics();
  const badge2 = `Δi_win (0→τ): ${fmt(wmBadge.dIwin)} A`;

  ctx.fillStyle = 'rgba(0,0,0,.22)';
  ctx.strokeStyle = 'rgba(30,42,58,.8)';
  roundRect(x0+10, y0+10, 330, 54, 10, true, true);

  ctx.fillStyle = 'rgba(151,166,178,.92)';
  ctx.fillText(badge1, x0+20, y0+30);

  ctx.fillStyle = 'rgba(178,123,255,.92)';
  ctx.fillText(badge2, x0+20, y0+48);

  // cursor dashed line + LAB overlays
  if(cursorActive && cursorX != null){
    const cx = clamp(cursorX, x0, x0 + w);

    // vertical line
    ctx.save();
    ctx.lineWidth = 1;
    ctx.setLineDash([6, 6]);
    ctx.strokeStyle = 'rgba(151,166,178,.55)';
    ctx.beginPath();
    ctx.moveTo(cx, y0);
    ctx.lineTo(cx, y0 + h);
    ctx.stroke();
    ctx.restore();

    // horizontal lines + tags (reuse ui.js computed cursor sample)
    if(cursorSample && cursorSample.active){
      const xRight = x0 + w;

      if($('#chV')?.checked){
        const yV = yMid - cursorSample.vsrc * Vscale;
        drawHLine(x0, xRight, yV, 'rgba(77,227,255,0.85)');
        drawTag(xRight, yV, `CH1 ${fmt(cursorSample.vsrc)} V`, 'rgba(77,227,255,0.95)', y0, y0 + h);
      }

      if($('#chI')?.checked){
        const yI = yMid - cursorSample.i * Iscale;
        drawHLine(x0, xRight, yI, 'rgba(255,210,77,0.85)');
        drawTag(xRight, yI, `CH2 ${fmt(cursorSample.i)} A`, 'rgba(255,210,77,0.92)', y0, y0 + h);
      }

      if($('#chVL')?.checked){
        const yVL = yMid - cursorSample.vL * VLscale;
        drawHLine(x0, xRight, yVL, 'rgba(178,123,255,0.80)');
        drawTag(xRight, yVL, `CH3 ${fmt(cursorSample.vL)} V`, 'rgba(178,123,255,0.90)', y0, y0 + h);
      }
      if($('#chVC')?.checked && cursorSample.vC != null){
        const yVC = yMid - cursorSample.vC * VCscale;
        drawHLine(x0, xRight, yVC, 'rgba(120,255,160,0.75)');
        drawTag(xRight, yVC, `CH4 ${fmt(cursorSample.vC)} V`, 'rgba(120,255,160,0.86)', y0, y0 + h);
      }
    }
  }

  ctx.restore();

  // ===== PERF: finish RENDER bucket =====
  if(perf?.toc) perf.toc();
}