// scope_draw.js
// Purpose: scope.js rendering helpers extracted to keep scope.js focused on frame orchestration.
// Behavior/numerics intentionally unchanged.

import { clamp } from "./dom.js";

export function makeScopeDraw(ctx, state){

function roundRect(x,y,w,h,r,fill,stroke){
  ctx.beginPath();
  const rr = Math.min(r, w/2, h/2);
  ctx.moveTo(x+rr, y);
  ctx.arcTo(x+w, y, x+w, y+h, rr);
  ctx.arcTo(x+w, y+h, x, y+h, rr);
  ctx.arcTo(x, y+h, x, y, rr);
  ctx.arcTo(x, y, x+w, y, rr);
  ctx.closePath();
  if(fill) ctx.fill();
  if(stroke) ctx.stroke();
}

function xFromU(u, x0, w){ return x0 + u*w; }

function fmtTimeAxis(t){
  const at = Math.abs(t);
  if (at < 1e-6) return `${(t*1e9).toFixed(0)}ns`;
  if (at < 1e-3) return `${(t*1e6).toFixed(1)}µs`;
  if (at < 1)    return `${(t*1e3).toFixed(2)}ms`;
  return `${t.toFixed(3)}s`;
}

function drawYAxisLabels({ x0, y0, w, h, divY }, perDiv, unit, color, offsetX = 0){
  if(!state.hold) return;
  ctx.save();
  ctx.font = '900 11px ui-sans-serif, system-ui';
  ctx.fillStyle = color;
  ctx.textAlign = 'right';       // important
  ctx.textBaseline = 'middle';

  const baseX = x0 + w - 8 - offsetX;  // right side

  for(let j=0;j<=divY;j++){
    const y = y0 + (j/divY)*h;
    const val = (divY/2 - j) * perDiv;
    ctx.fillText(`${val.toFixed(2)} ${unit}`, baseX, y);
  }

  ctx.restore();
}

function drawXAxisLabels({ x0, y0, w, h, divX }, tStart, span){
  if(!state.hold) return;
  ctx.save();
  ctx.font = '900 11px ui-sans-serif, system-ui';
  ctx.fillStyle = 'rgba(151,166,178,.85)';
  ctx.textAlign = 'center';
  ctx.textBaseline = 'top';

  const y = y0 + h + 8;
  for(let i=0;i<=divX;i++){
    const x = x0 + (i/divX)*w;
    const t = tStart + (i/divX)*span;
    ctx.fillText(fmtTimeAxis(t), x, y);
  }
  ctx.restore();
}

function drawTrace(samples, x0, yMid, scale, stroke, lw, w){
  ctx.save();
  ctx.strokeStyle = stroke;
  ctx.lineWidth = lw;
  ctx.beginPath();
  for(let i=0;i<samples.length;i++){
    const u = i/(samples.length-1);
    const x = xFromU(u, x0, w);
    const y = yMid - samples[i]*scale;
    if(i===0) ctx.moveTo(x,y); else ctx.lineTo(x,y);
  }
  ctx.stroke();
  ctx.restore();
}

function drawHLine(x0, x1, y, color){
  ctx.save();
  ctx.lineWidth = 1;
  ctx.setLineDash([5,5]);
  ctx.strokeStyle = color;
  ctx.beginPath();
  ctx.moveTo(x0, y);
  ctx.lineTo(x1, y);
  ctx.stroke();
  ctx.restore();
}

function drawTag(xRight, y, text, color, yTop, yBot){
  ctx.save();
  ctx.font = '900 11px ui-sans-serif, system-ui';
  const padX = 8;
  const tw = ctx.measureText(text).width;
  const w = tw + padX*2;
  const h = 20;

  const x = xRight - w - 10;
  const yClamped = clamp(y - h/2, yTop + 6, yBot - h - 6);

  ctx.fillStyle = 'rgba(0,0,0,.55)';
  ctx.strokeStyle = color;
  roundRect(x, yClamped, w, h, 7, true, true);

  ctx.fillStyle = color;
  ctx.fillText(text, x + padX, yClamped + 14);
  ctx.restore();
}


// --- robustness helpers (prevents NaN autoscale killing traces) ---
function maxAbsFinite(arr, fallback){
  let m = 0;
  for(let i=0;i<arr.length;i++){
    const v = arr[i];
    if(!Number.isFinite(v)) continue;
    const a = Math.abs(v);
    if(a > m) m = a;
  }
  return (m > 0) ? m : (fallback ?? 0);
}

function sanitizeInPlace(arr){
  // replace non-finite samples with last finite value (keeps trace continuous)
  let last = 0;
  for(let i=0;i<arr.length;i++){
    const v = arr[i];
    if(Number.isFinite(v)) last = v;
    else arr[i] = last;
  }
}

  // expose helpers (same names as in legacy scope.js)
  return {
    roundRect,
    xFromU,
    fmtTimeAxis,
    drawYAxisLabels,
    drawXAxisLabels,
    drawTrace,
    drawHLine,
    drawTag,
    maxAbsFinite,
    sanitizeInPlace
  };
}