// scope.js
import { $, clamp, fmtDivMs, fmt } from "./dom.js";
import { state } from "./state.js";
import { tau, periodPulse, periodSine } from "./core/derived.js";
import { buildWaves } from "./wave_engine.js";
import { makeScopeDraw } from "./scope_draw.js";
import { impulse_abs_vL_0_to, peak_abs_vL_0_to, windowMagneticMetrics } from "./metrics.js";
import {
  updateMeasStrip,
  updateMeter,
  updatePkVLmeter,
  updatePowerPanel,
  updateDerived,
  updatePeriodicityBadge
} from "./panels.js";

let scope = null;
let ctx = null;

// drawing helpers (bound to ctx in initScope; extracted into scope_draw.js)
let roundRect = null;
let drawYAxisLabels = null;
let drawXAxisLabels = null;
let drawTrace = null;
let drawTag = null;
let maxAbsFinite = null;
let sanitizeInPlace = null;

// last rendered scope data for cursor readout
let lastScopeData = null;

// autoscale smoothing factor (scope vertical stability)
const AUTOSCALE_ALPHA = 0.10;

// smoothed autoscale (prevents flicker)
let Vfs_s = null;
let iPeak_s = null;
let vLPeak_s = null;
let vCPeak_s = null;

// cursor state (for dashed line)
let cursorX = null;       // CSS px inside canvas
let cursorActive = false;

// --- HOLD caching (keep FREEZE interactive but cheap) ---
let _holdKey = "";
let _holdBuilt = null;     // cached buildWaves result
let _holdScales = null;    // cached autoscale results

// cursor sample (computed in ui.js, reused for LAB overlays)
let cursorSample = null; // { active, vsrc, i, vL, vR, vC? }
export function setCursorSample(s){ cursorSample = s || null; }
export function clearCursorSample(){ cursorSample = null; }

// running window timebase (scope "RUN/STOP")
let runT_s = 0;          // current left-edge time of the displayed window (seconds)
let lastNow_s = null;    // for dt integration

// HUD-safe text node (prevents wiping injected badges/spans)
let hudRightTextEl = null;

export function initScope(canvasEl){
  scope = canvasEl;
  if(!scope) throw new Error("initScope: canvas element missing");
  ctx = scope.getContext('2d', { alpha:true, desynchronized:true });

  ({
    roundRect,
    drawYAxisLabels,
    drawXAxisLabels,
    drawTrace,
    drawTag,
    maxAbsFinite,
    sanitizeInPlace
  } = makeScopeDraw(ctx, state));

  resizeScope();
}

export function resizeScope(){
  if(!scope) return;
  const dpr = Math.min(2, window.devicePixelRatio || 1);
  const rect = scope.getBoundingClientRect();
  const w = Math.max(1, Math.floor(rect.width * dpr));
  const h = Math.max(1, Math.floor(rect.height * dpr));
  scope.width = w;
  scope.height = h;
  scope.dataset.dpr = String(dpr);
}

export function getLastScopeData(){ return lastScopeData; }

export function setCursorState(xCssPx, active){
  cursorX = xCssPx;
  cursorActive = !!active;
}

export function clearCursorState(){
  cursorX = null;
  cursorActive = false;
}

function drawCursorLine(x, y0, y1){
  // pure rendering; no numeric coupling
  ctx.save();
  ctx.setLineDash([6,6]);
  ctx.strokeStyle = 'rgba(255,255,255,.18)';
  ctx.lineWidth = 1;
  ctx.beginPath();
  ctx.moveTo(x, y0);
  ctx.lineTo(x, y1);
  ctx.stroke();
  ctx.restore();
}

export function renderScope(){
  if(!scope || !ctx) return;

  const perf = window.__perf || null;

  // =========================
  // RENDER: setup / grid / clear
  // =========================
  if(perf?.tic) perf.tic("render");

  const dpr = parseFloat(scope.dataset.dpr || '1');
  const W = scope.width / dpr;
  const H = scope.height / dpr;
  ctx.setTransform(dpr,0,0,dpr,0,0);
  ctx.clearRect(0,0,W,H);

  const pad = 22;
  const x0 = pad, y0 = pad+18, w = W - 2*pad, h = H - 2*pad - 22;
  const divX = 10, divY = 8;
  const yMid = y0 + h/2;

  ctx.save();
  ctx.strokeStyle = 'rgba(30,42,58,.9)';
  ctx.lineWidth = 1;
  roundRect(x0, y0, w, h, 14, false, true);
  ctx.restore();

  ctx.save();
  for(let i=1;i<divX;i++){
    const x = x0 + (i/divX)*w;
    ctx.strokeStyle = (i===5) ? 'rgba(151,166,178,.28)' : 'rgba(30,42,58,.55)';
    ctx.beginPath(); ctx.moveTo(x,y0); ctx.lineTo(x,y0+h); ctx.stroke();
  }
  for(let j=1;j<divY;j++){
    const y = y0 + (j/divY)*h;
    ctx.strokeStyle = (j===4) ? 'rgba(151,166,178,.28)' : 'rgba(30,42,58,.55)';
    ctx.beginPath(); ctx.moveTo(x0,y); ctx.lineTo(x0+w,y); ctx.stroke();
  }
  ctx.restore();

  if(perf?.toc) perf.toc();

  const span = (state.timeDiv_ms*1e-3) * 10;
  const tt = tau();

  const N = Math.max(800, Math.min(4000, Math.floor(w)));
  const dt = span/(N-1);

  // =========================
  // PHYSICS: timebase + buildWaves (cached in HOLD) + autoscale
  // =========================
  if(perf?.tic) perf.tic("physics");

  const now_s = performance.now() * 1e-3;
  if(lastNow_s === null) lastNow_s = now_s;

  if(!state.hold){
    const dtRun = Math.min(0.05, Math.max(0, now_s - lastNow_s));
    runT_s += dtRun;
  }
  lastNow_s = now_s;

  const tStart = runT_s - 0.2 * span;

  const topo = (state.topology || "seriesRLC");
  const key = [
    state.srcMode, topo,
    state.V, state.f, state.fs, state.dutyPct,
    state.L, state.Rcoil, state.Rload, state.C,
    state.timeDiv_ms,
    N,
    tStart, span
  ].join("|");

  let built;
  let scales;

  if(state.hold && _holdBuilt && _holdScales && key === _holdKey){
    built = _holdBuilt;
    scales = _holdScales;
  } else {
    built = buildWaves({ tStart, span, N });

    sanitizeInPlace(built.vSrc);
    sanitizeInPlace(built.i);
    sanitizeInPlace(built.vL);
    if(built.vC) sanitizeInPlace(built.vC);

    const vS0  = built.vSrc;
    const iS0  = built.i;
    const vL0  = built.vL;
    const vC0  = built.vC || null;

    const Vfs_t    = Math.max(1,    maxAbsFinite(vS0, 1)  * 1.15);
    const iPeak_t  = Math.max(0.01, maxAbsFinite(iS0, 0.01));
    const vLPeak_t = Math.max(1,    maxAbsFinite(vL0, 1));
    const vCPeak_t = Math.max(1,    vC0 ? maxAbsFinite(vC0, 1) : 1);

    let a = AUTOSCALE_ALPHA;
    try {
      const pq = window.__lastPQ || null;
      if (pq && isFinite(pq.phiDeg)) {
        const phaseRad = (pq.phiDeg || 0) * Math.PI/180;
        const proximity = Math.exp(-Math.abs(phaseRad) * 8);
        a = AUTOSCALE_ALPHA + 0.35 * proximity;
      }
    } catch(_){}

    if(Vfs_s   === null) Vfs_s   = Vfs_t;
    if(iPeak_s === null) iPeak_s = iPeak_t;
    if(vLPeak_s=== null) vLPeak_s= vLPeak_t;
    if(vCPeak_s=== null) vCPeak_s= vCPeak_t;

    Vfs_s    = Vfs_s    + a*(Vfs_t    - Vfs_s);
    iPeak_s  = iPeak_s  + a*(iPeak_t  - iPeak_s);
    vLPeak_s = vLPeak_s + a*(vLPeak_t - vLPeak_s);
    vCPeak_s = vCPeak_s + a*(vCPeak_t - vCPeak_s);

    const Vfs    = Vfs_s;
    const iPeak  = iPeak_s;
    const vLPeak = vLPeak_s;
    const vCPeak = vCPeak_s;

    const VperDiv  = Vfs/4;
    const Vscale   = h/(divY*(Vfs/4));

    const IperDiv  = Math.max(1e-6, iPeak/3);
    const Iscale   = h/(divY*IperDiv);

    const VLperDiv = Math.max(1e-6, vLPeak/3);
    const VLscale  = h/(divY*VLperDiv);

    const VCperDiv = Math.max(1e-6, vCPeak/3);
    const VCscale  = h/(divY*VCperDiv);

    scales = { VperDiv, Vscale, IperDiv, Iscale, VLperDiv, VLscale, VCperDiv, VCscale };

    if(state.hold){
      _holdKey = key;
      _holdBuilt = built;
      _holdScales = scales;
    } else {
      _holdKey = "";
      _holdBuilt = null;
      _holdScales = null;
    }
  }

  if(perf?.toc) perf.toc();

  const vS  = built.vSrc;
  const iS  = built.i;
  const vLS = built.vL;

  if(!renderScope._zeroVC || renderScope._zeroVC.length !== N){
    renderScope._zeroVC = new Array(N).fill(0);
  }
  const vCS = built.vC || renderScope._zeroVC;

  const { VperDiv, Vscale, IperDiv, Iscale, VLperDiv, VLscale, VCperDiv, VCscale } = scales;

  // =========================
  // RENDER: traces + HUD + panels
  // =========================
  if(perf?.tic) perf.tic("render");

  const frame = {
    time: { tStart, span, dt, tau: tt },
    geom: { x0, y0, w, h, yMid },
    waves: { vSrc: vS, i: iS, vL: vLS, vC: vCS },
    scale: { Vscale, Iscale, VLscale, VCscale }
  };
  lastScopeData = frame;

  // traces
  if($('#chV')?.checked)  drawTrace(vS,  x0, yMid, Vscale,  'rgba(77,227,255,0.92)', 2.0, w);
  if($('#chI')?.checked)  drawTrace(iS,  x0, yMid, Iscale,  'rgba(255,210,77,0.92)', 2.0, w);
  if($('#chVL')?.checked) drawTrace(vLS, x0, yMid, VLscale, 'rgba(178,123,255,0.86)', 2.0, w);
  if($('#chVC')?.checked) drawTrace(vCS, x0, yMid, VCscale, 'rgba(120,255,160,0.86)', 2.0, w);

  //cursor vertical trace (restored)
  if(cursorActive && cursorX != null){
    const x = clamp(cursorX, x0, x0 + w);
    drawCursorLine(x, y0, y0 + h);

    // === horizontal cursor traces (per enabled channel, channel-colored) ===
    ctx.save();
    ctx.setLineDash([6,6]);
    ctx.lineWidth = 1;

    function hline(y, color){
      const yy = clamp(y, y0, y0 + h);
      ctx.strokeStyle = color;
      ctx.beginPath();
      ctx.moveTo(x0, yy);
      ctx.lineTo(x0 + w, yy);
      ctx.stroke();
    }

    if(cursorSample && cursorSample.active){
      if($('#chV')?.checked)
        hline(yMid - cursorSample.vsrc * Vscale, 'rgba(77,227,255,0.55)');

      if($('#chI')?.checked)
        hline(yMid - cursorSample.i * Iscale, 'rgba(255,210,77,0.55)');

      if($('#chVL')?.checked)
        hline(yMid - cursorSample.vL * VLscale, 'rgba(178,123,255,0.55)');

      if($('#chVC')?.checked && cursorSample.vC != null)
        hline(yMid - cursorSample.vC * VCscale, 'rgba(120,255,160,0.55)');
    }

    ctx.restore();

    // cursor tags (if we have sample)
    if(cursorSample && cursorSample.active){
      const yTop = y0, yBot = y0 + h;
      if($('#chV')?.checked)  drawTag(x0+w, yMid - cursorSample.vsrc*Vscale, `v ${fmt(cursorSample.vsrc)}V`, 'rgba(77,227,255,0.92)', yTop, yBot);
      if($('#chI')?.checked)  drawTag(x0+w, yMid - cursorSample.i*Iscale,    `i ${fmt(cursorSample.i)}A`,    'rgba(255,210,77,0.92)', yTop, yBot);
      if($('#chVL')?.checked) drawTag(x0+w, yMid - cursorSample.vL*VLscale,  `vL ${fmt(cursorSample.vL)}V`, 'rgba(178,123,255,0.86)', yTop, yBot);
      if($('#chVC')?.checked && cursorSample.vC != null)
        drawTag(x0+w, yMid - cursorSample.vC*VCscale, `vC ${fmt(cursorSample.vC)}V`, 'rgba(120,255,160,0.86)', yTop, yBot);
    }
  }

  // axes (only in HOLD)
  if(state.hold){
    let axisOffset = 0;
    if($('#chV')?.checked){  drawYAxisLabels({ x0, y0, w, h, divY }, VperDiv,  'V', 'rgba(77,227,255,0.90)', axisOffset); axisOffset += 85; }
    if($('#chI')?.checked){  drawYAxisLabels({ x0, y0, w, h, divY }, IperDiv,  'A', 'rgba(255,210,77,0.90)', axisOffset); axisOffset += 85; }
    if($('#chVL')?.checked){ drawYAxisLabels({ x0, y0, w, h, divY }, VLperDiv, 'V', 'rgba(178,123,255,0.86)', axisOffset); axisOffset += 85; }
    if($('#chVC')?.checked){ drawYAxisLabels({ x0, y0, w, h, divY }, VCperDiv, 'V', 'rgba(120,255,160,0.86)', axisOffset); axisOffset += 85; }
    drawXAxisLabels({ x0, y0, w, h, divX }, tStart, span);
  }

  // ---- HUD (SAFE: do not overwrite badge spans) ----
  const modeTxt = (state.srcMode === 'sine')
    ? `Sine: f=${state.fs.toFixed(0)} Hz`
    : `Pulse: f=${state.f.toFixed(0)} Hz · Duty=${state.dutyPct.toFixed(2)}%`;

  const status = state.hold ? "FROZEN" : "LIVE";
  const runDot = state.hold ? "■" : (Math.floor(now_s * 2) % 2 ? "●" : " ");

  const chBits = [];
  if($('#chV')?.checked)  chBits.push(`CH1 ${VperDiv.toFixed(2)} V/div`);
  if($('#chI')?.checked)  chBits.push(`CH2 ${IperDiv.toFixed(3)} A/div`);
  if($('#chVL')?.checked) chBits.push(`CH3 ${VLperDiv.toFixed(2)} V/div`);
  if($('#chVC')?.checked) chBits.push(`CH4 ${VCperDiv.toFixed(2)} V/div`);

  const chTxt = (state.hold && chBits.length) ? `  |  ${chBits.join('  ·  ')}` : "";

  const hudRight = $('#hudRight') || document.querySelector('.hudRight');
  if (hudRight) {
    if (!hudRightTextEl || hudRightTextEl.parentNode !== hudRight) {
      hudRightTextEl = hudRight.querySelector('.hudRightText');
      if (!hudRightTextEl) {
        hudRightTextEl = document.createElement('span');
        hudRightTextEl.className = 'hudRightText';
        hudRight.insertBefore(hudRightTextEl, hudRight.firstChild || null);
      }
    }

    hudRightTextEl.textContent =
      `${runDot} ${status}  |  ${modeTxt}  |  Time: ${fmtDivMs(state.timeDiv_ms)}${chTxt}  |  τ = ${(tt*1e3).toFixed(3)} ms`;
  }

  // panels + KPIs
  updateMeasStrip();

  const nowImpulse = impulse_abs_vL_0_to(tt);
  updateMeter(nowImpulse);

  const nowPkVL = peak_abs_vL_0_to(tt);
  updatePkVLmeter(nowPkVL);

  updatePowerPanel();
  updateDerived();

  // periodicity badge (after HUD text so it can append/modify without being clobbered)
  updatePeriodicityBadge();

  // canvas footer badges
  ctx.save();
  ctx.font = '700 12px ui-sans-serif, system-ui';

  const badge1 = `∫|vL|dt (0→τ): ${nowImpulse.toExponential(2)} V·s`;
  const wmBadge = windowMagneticMetrics();
  const badge2 = `Δi_win (0→τ): ${fmt(wmBadge.dIwin)} A`;

  ctx.fillStyle = 'rgba(255,255,255,.12)';
  ctx.fillText(badge1, x0 + 10, y0 + h + 18);
  ctx.fillText(badge2, x0 + 10, y0 + h + 36);
  ctx.restore();

  if(perf?.toc) perf.toc();
}