// app/rlc_sine.js
import { state } from "./state.js";
import { Rtotal } from "./core/derived.js";

// Series RLC phasor parameters for sine excitation.
// Convention:
//   v(t) = Vpk sin(ωt)
//   i(t) = Ipk sin(ωt - φ)
// φ > 0 => current lags (inductive), φ < 0 => current leads (capacitive)
export function rlcSineParamsSeries(){
  const eps = 1e-18;

  const R = Rtotal();
  const L = Math.max(0, state.L);
  const C = Math.max(0, state.C || 0);

  const f = Math.max(eps, state.fs);
  const w = 2 * Math.PI * f;

  const Xl = w * L;
  const Xc = (C > 0) ? (1 / Math.max(eps, w * C)) : Infinity;

  // Net reactance
  const X = (C > 0) ? (Xl - Xc) : Xl;

  const Zmag = Math.sqrt(R*R + X*X);
  const phi  = Math.atan2(X, R);

  const Vpk  = state.V;
  const Vrms = Vpk / Math.SQRT2;

  const Ipk  = (Zmag > eps) ? (Vpk / Zmag) : 0;
  const Irms = Ipk / Math.SQRT2;

  // Useful helpers
  const f0 = (L > 0 && C > 0) ? (1 / (2*Math.PI*Math.sqrt(L*C))) : 0;

  return { R, L, C, f, w, Xl, Xc, X, Zmag, phi, Vpk, Vrms, Ipk, Irms, f0 };
}

// Parallel: (R+L in series) || C
// Convention:
//   v(t) = Vpk sin(ωt)
//   i_total(t) = Ipk sin(ωt - φ)   where φ = arg(Zeq)
// Notes:
//  - vC(t) = v(t) (same node voltage)
//  - vL(t) here is the inductor voltage INSIDE the RL branch (useful/physical)
export function rlcSineParamsParallelRL(){
  const eps = 1e-18;

  const R = Rtotal();
  const L = Math.max(0, state.L);
  const C = Math.max(0, state.C || 0);

  const f = Math.max(eps, state.fs);
  const w = 2 * Math.PI * f;

  const Xl = w * L;
  const Bc = (C > 0) ? (w * C) : 0; // susceptance magnitude

  // RL branch impedance: Zrl = R + jXl
  const Zrl_mag2 = R*R + Xl*Xl;
  const Zrl_mag  = Math.sqrt(Math.max(eps, Zrl_mag2));
  const phiRL    = Math.atan2(Xl, R); // arg(Zrl), inductive lag

  // Admittance:
  // Yrl = 1/(R + jXl) = (R - jXl)/(R^2 + Xl^2) = G + jBrl
  const G   = R / Math.max(eps, Zrl_mag2);
  const Brl = -Xl / Math.max(eps, Zrl_mag2);

  // Capacitor admittance: Yc = jωC = jBc
  const B = Brl + Bc;

  // Y = G + jB  =>  Z = 1/Y
  const Ymag = Math.sqrt(G*G + B*B);
  const Zmag = (Ymag > eps) ? (1 / Ymag) : 1/eps;

  // arg(Z) = -arg(Y)
  const phi = -Math.atan2(B, G);

  const Vpk  = state.V;
  const Vrms = Vpk / Math.SQRT2;

  const Ipk  = (Zmag > eps) ? (Vpk / Zmag) : 0;
  const Irms = Ipk / Math.SQRT2;

  // Branch currents (peak) (useful for vL rendering)
  const Irl_pk = (Zrl_mag > eps) ? (Vpk / Zrl_mag) : 0;
  const Ic_pk  = Vpk * Bc; // |I| = V * ωC

  return {
    R, L, C, f, w,
    Xl, Bc,
    G, B,            // admittance parts
    Zmag, phi,       // equivalent impedance magnitude + phase
    Vpk, Vrms,
    Ipk, Irms,
    Irl_pk, phiRL,   // RL-branch current (peak) + its lag
    Ic_pk            // capacitor branch current (peak)
  };
}