// app/rlc_pulse.js
// Series RLC pulse-train (rectangular) steady-periodic solution.
// Convention:
//   vsrc(t) = V  for 0<=t<Ton, and 0 for Ton<=t<T
// State vector:
//   x = [ i, vC ]^T
// Dynamics:
//   di/dt  = (vsrc - R*i - vC)/L
//   dvC/dt = i/C
//
// Implementation notes:
// - We solve the linear 2-state system exactly per segment using a 2x2 matrix exponential.
// - Periodic steady state is enforced by solving (I - Φoff Φon) x0 = Φoff Γon V.
// - Wave outputs use KVL closure: vL = vsrc - R*i - vC.
//
// PLUS:
// - Parallel pulse mode: (R+L) branch in parallel with C across an ideal Vsrc.
//   Uses a tiny trapezoid edge time to avoid dv/dt impulses in iC=C·dv/dt.

import { state } from "./state.js";
import { Rtotal, periodPulse, duty } from "./core/derived.js";

const EPS = 1e-18;

// ---------- 2x2 helpers ----------
function mat2_mul(A, B){
  // A,B: [a,b,c,d] representing [[a,b],[c,d]]
  return [
    A[0]*B[0] + A[1]*B[2],
    A[0]*B[1] + A[1]*B[3],
    A[2]*B[0] + A[3]*B[2],
    A[2]*B[1] + A[3]*B[3],
  ];
}
function mat2_add(A,B){ return [A[0]+B[0],A[1]+B[1],A[2]+B[2],A[3]+B[3]]; }
function mat2_sub(A,B){ return [A[0]-B[0],A[1]-B[1],A[2]-B[2],A[3]-B[3]]; }
function mat2_scale(A,s){ return [A[0]*s,A[1]*s,A[2]*s,A[3]*s]; }
function mat2_eye(){ return [1,0,0,1]; }
function mat2_det(A){ return A[0]*A[3] - A[1]*A[2]; }
function mat2_inv(A){
  const d = mat2_det(A);
  if(Math.abs(d) < 1e-30) return null;
  const invd = 1/d;
  return [ A[3]*invd, -A[1]*invd, -A[2]*invd, A[0]*invd ];
}
function mat2_vec(A, v){
  return [ A[0]*v[0] + A[1]*v[1], A[2]*v[0] + A[3]*v[1] ];
}
function solve2(M, b){
  // Solve M x = b for 2x2
  const d = mat2_det(M);
  if(Math.abs(d) < 1e-30) return [0,0];
  const invd = 1/d;
  return [
    ( b[0]*M[3] - M[1]*b[1] )*invd,
    ( M[0]*b[1] - b[0]*M[2] )*invd
  ];
}

// expm for 2x2 via trace/determinant closed form.
// Returns Φ(t) as [a,b,c,d].
function makeExpm2x2(A){
  const tr = A[0] + A[3];
  const det = mat2_det(A);
  const halfTr = 0.5*tr;

  // B = A - (tr/2)I
  const B = [A[0]-halfTr, A[1], A[2], A[3]-halfTr];

  const disc = tr*tr - 4*det; // can be negative (underdamped)

  // precompute to avoid branches in hot loops
  const useImag = (disc < 0);
  const absDelta = Math.sqrt(Math.abs(disc));

  return function phi(t){
    const et = Math.exp(halfTr * t);

    // Handle near-critical (delta ~ 0) with series to avoid 0/0
    if(absDelta < 1e-12){
      // exp(A t) ≈ e^{halfTr t} ( I + t B )
      const Bt = mat2_scale(B, t);
      return mat2_scale(mat2_add(mat2_eye(), Bt), et);
    }

    const h = 0.5*absDelta*t;
    let c, s_over_delta;

    if(useImag){
      // delta = i*absDelta
      c = Math.cos(h);
      // (2/delta)*sinh(delta t/2) => 2*sin(absDelta t/2)/absDelta
      s_over_delta = (2*Math.sin(h))/absDelta;
    }else{
      c = Math.cosh(h);
      // (2/delta)*sinh(delta t/2)
      s_over_delta = (2*Math.sinh(h))/absDelta;
    }

    // Φ = e^{tr t/2} [ c I + s_over_delta * B ]
    const term1 = mat2_scale(mat2_eye(), c);
    const term2 = mat2_scale(B, s_over_delta);
    return mat2_scale(mat2_add(term1, term2), et);
  };
}

function makeSegmentSolver({R,L,C}){
  // A x + b u
  // A = [[-R/L, -1/L],[1/C, 0]]
  const invL = 1/Math.max(EPS, L);
  const invC = 1/Math.max(EPS, C);

  const A = [
    -R*invL,   -invL,
     invC,      0
  ];
  const invA = mat2_inv(A) || [0,0,0,0]; // should be invertible for normal params
  const phi = makeExpm2x2(A);
  const b = [invL, 0];

  function gamma(t){
    // Γ = A^{-1} (Φ - I) b
    const PHI = phi(t);
    const PHI_minus_I = mat2_sub(PHI, mat2_eye());
    const tmp = mat2_vec(PHI_minus_I, b);
    return mat2_vec(invA, tmp);
  }

  return { A, phi, gamma };
}

export function rlcPulseSteadyState(){
  const R = Rtotal();
  const L = Math.max(0, state.L);
  const C = Math.max(0, state.C || 0);

  const T = periodPulse();
  const Ton = T * duty();
  const Toff = Math.max(0, T - Ton);

  const V = state.V;

  // Guard: if no C or invalid L, fall back to trivial
  if(!(C > 0) || !(L > 0) || T <= 0){
    return {
      R,L,C,T,Ton,Toff,V,
      x0:[0,0], x1:[0,0],
      vSrcAt:(t)=> (t < Ton ? V : 0),
      stateAt:(t)=> [0,0],
      iAt:(t)=> 0,
      vCAt:(t)=> 0,
      vLAt:(t)=> (t < Ton ? V : 0),
    };
  }

  const on  = makeSegmentSolver({R,L,C});
  const off = makeSegmentSolver({R,L,C});

  const Phi_on_Ton  = on.phi(Ton);
  const Gam_on_Ton  = on.gamma(Ton);

  const Phi_off_Toff = off.phi(Toff);

  // x1 = Φon(Ton) x0 + Γon(Ton) V
  // x0 = Φoff(Toff) x1
  // => (I - Φoff Φon) x0 = Φoff Γon V
  const Phi_off_on = mat2_mul(Phi_off_Toff, Phi_on_Ton);
  const M = mat2_sub(mat2_eye(), Phi_off_on);

  const rhs = mat2_vec(Phi_off_Toff, [Gam_on_Ton[0]*V, Gam_on_Ton[1]*V]);
  const x0 = solve2(M, rhs);

  const x1 = (()=>{
    const term = mat2_vec(Phi_on_Ton, x0);
    return [ term[0] + Gam_on_Ton[0]*V, term[1] + Gam_on_Ton[1]*V ];
  })();

  function vSrcAt(tm){ return (tm < Ton) ? V : 0; }

  function stateAt(tm){
    if(tm < Ton){
      const PHI = on.phi(tm);
      const GAM = on.gamma(tm);
      const term = mat2_vec(PHI, x0);
      return [ term[0] + GAM[0]*V, term[1] + GAM[1]*V ];
    }else{
      const t2 = tm - Ton;
      const PHI = off.phi(t2);
      return mat2_vec(PHI, x1);
    }
  }

  function iAt(tm){ return stateAt(tm)[0]; }
  function vCAt(tm){ return stateAt(tm)[1]; }
  function vLAt(tm){
    const is = iAt(tm);
    const vc = vCAt(tm);
    const vs = vSrcAt(tm);
    return vs - R*is - vc;
  }

  return { R,L,C,T,Ton,Toff,V, x0, x1, vSrcAt, stateAt, iAt, vCAt, vLAt };
}

export function rlcPulseWave({ tStart, span, N }){
  const dt = span / Math.max(1, (N - 1));
  const vSrc = new Array(N);
  const i    = new Array(N);
  const vL   = new Array(N);
  const vC   = new Array(N);

  const sol = rlcPulseSteadyState();

  const T = sol.T || periodPulse();

  for(let k=0;k<N;k++){
    const t = tStart + k*dt;
    const tm = ((t % T) + T) % T;

    const vs = sol.vSrcAt(tm);
    const st = sol.stateAt(tm);

    const is = st[0];
    const vc = st[1];
    const vl = vs - sol.R*is - vc;

    vSrc[k] = vs;
    i[k]    = is;
    vC[k]   = vc;
    vL[k]   = vl;
  }

  return { dt, vSrc, i, vL, vC, hasVC: true };
}

// ---------------------------------------------------------------------------
// Parallel pulse mode:  (R+L) branch in parallel with C across an ideal Vsrc.
// Conventions for rendering (matches sine-parallel behavior in wave_engine.js):
//   vC(t) = vSrc(t)
//   i(t)  = i_total(t) = i_RL(t) + i_C(t)
//   vL(t) = inductor voltage INSIDE the RL branch
// Notes:
// - With ideal step edges, i_C = C dv/dt contains impulses. We avoid that by
//   using a very small trapezoid edge time (finite dv/dt).
// - RL branch is solved exactly for piecewise: linear ramp / constant.
// ---------------------------------------------------------------------------

function clamp(x, a, b){ return Math.max(a, Math.min(b, x)); }

function _pulseEdgeTimes(T, Ton){
  // finite edges to avoid dv/dt impulses
  // default: 0.2% of period, bounded.
  const base = clamp(0.002*T, 50e-9, 0.05*Math.max(1e-12, Ton));
  const Tr = base;
  const Tf = base;
  return { Tr, Tf };
}

function _mapConst(i0, vConst, dt, tt, R){
  const A = Math.exp(-dt / Math.max(1e-18, tt));
  const Iinf = (R > 0) ? (vConst / R) : 0;
  const i1 = Iinf + (i0 - Iinf)*A;
  const B = i1 - A*i0;
  return { A, B, i1 };
}

function _mapRamp(i0, v0, v1, dt, tt, R, L){
  // v(t) = v0 + k t,  0..dt,  k=(v1-v0)/dt
  // i(t) = e^{-t/tt} [ i0 + (1/L) ∫_0^t e^{u/tt} v(u) du ]
  const ttn = Math.max(1e-18, tt);
  const A = Math.exp(-dt / ttn);
  const k = (dt > 0) ? ((v1 - v0) / dt) : 0;

  // Integral at t=dt:
  // ∫ e^{u/tt}(v0 + k u) du = v0*tt(e^{dt/tt}-1) + k[ tt(dt-tt)e^{dt/tt} + tt^2 ]
  const e = Math.exp(dt / ttn);
  const I = (v0*ttn*(e - 1)) + (k * (ttn*(dt - ttn)*e + ttn*ttn));
  const i1 = A * (i0 + (I / Math.max(1e-18, L)));
  const B = i1 - A*i0;
  return { A, B, i1 };
}

export function rlcPulseParallelSteadyState(){
  const R = Rtotal();
  const L = Math.max(0, state.L);
  const C = Math.max(0, state.C || 0);

  const T = periodPulse();
  const Ton = T * duty();
  const Toff = Math.max(0, T - Ton);
  const V = state.V;

  // Guards
  if(!(T > 0) || !(L > 0) || !(R > 0) || !(C > 0)){
    return {
      R,L,C,T,Ton,Toff,V,
      Tr:0, Tf:0, Th:0, tt: (R>0?L/R:0),
      i0:0,
      vSrcAt:(t)=> (t < Ton ? V : 0),
      dvSrcAt:(t)=> 0,
      iRLAt:(t)=> 0,
      iCAt:(t)=> 0,
      iAt:(t)=> 0,
      vCAt:(t)=> (t < Ton ? V : 0),
      vLAt:(t)=> (t < Ton ? V : 0),
    };
  }

  const tt = L / R;

  const { Tr, Tf } = _pulseEdgeTimes(T, Ton);
  const TrEff = Math.min(Tr, Math.max(0, Ton));
  const TfEff = Math.min(Tf, Math.max(0, Ton - TrEff));
  const Th = Math.max(0, Ton - TrEff - TfEff);

  // Segment list for RL branch in one period
  const segs = [];
  if(TrEff > 0) segs.push({ kind:'ramp', dt:TrEff, v0:0, v1:V });
  if(Th > 0)    segs.push({ kind:'const', dt:Th,  v:V });
  if(TfEff > 0) segs.push({ kind:'ramp', dt:TfEff, v0:V, v1:0 });
  if(Toff > 0)  segs.push({ kind:'const', dt:Toff, v:0 });

  // Full-period map: iT = Aall*i0 + Ball  => i0 = Ball/(1-Aall)
  let Aall = 1;
  let Ball = 0;
  for(const s of segs){
    const m = (s.kind === 'const')
      ? _mapConst(0, s.v, s.dt, tt, R)
      : _mapRamp(0, s.v0, s.v1, s.dt, tt, R, L);
    Aall = m.A * Aall;
    Ball = m.A * Ball + m.B;
  }

  const denom = (1 - Aall);
  const i0 = (Math.abs(denom) < 1e-18) ? 0 : (Ball / denom);

  // Segment start currents
  const segStarts = [];
  let cur = 0;
  let iStart = i0;
  for(const s of segs){
    segStarts.push({ t0: cur, i0: iStart, seg: s });
    const out = (s.kind === 'const')
      ? _mapConst(iStart, s.v, s.dt, tt, R).i1
      : _mapRamp(iStart, s.v0, s.v1, s.dt, tt, R, L).i1;
    iStart = out;
    cur += s.dt;
  }

  function vSrcAt(tm){
    if(tm < 0) return 0;
    if(tm >= T) tm = ((tm % T) + T) % T;
    if(tm >= Ton) return 0;

    if(TrEff > 0 && tm < TrEff) return V * (tm / TrEff);

    const t2 = tm - TrEff;
    if(t2 < Th) return V;

    const t3 = t2 - Th;
    if(TfEff > 0 && t3 < TfEff) return V * (1 - (t3 / TfEff));

    return 0;
  }

  function dvSrcAt(tm){
    if(tm < 0) return 0;
    if(tm >= T) tm = ((tm % T) + T) % T;
    if(tm >= Ton) return 0;

    if(TrEff > 0 && tm < TrEff) return V / TrEff;

    const t2 = tm - TrEff;
    if(t2 < Th) return 0;

    const t3 = t2 - Th;
    if(TfEff > 0 && t3 < TfEff) return -V / TfEff;

    return 0;
  }

  function iRLAt(tm){
    if(tm < 0) return i0;
    if(tm >= T) tm = ((tm % T) + T) % T;

    for(const s0 of segStarts){
      const { t0, i0:ii0, seg } = s0;
      const t1 = t0 + seg.dt;
      if(tm >= t0 && tm < t1){
        const dtp = tm - t0;
        if(seg.kind === 'const'){
          const A = Math.exp(-dtp / Math.max(1e-18, tt));
          const Iinf = seg.v / R;
          return Iinf + (ii0 - Iinf)*A;
        }else{
          const ttn = Math.max(1e-18, tt);
          const A = Math.exp(-dtp / ttn);
          const k = (seg.dt > 0) ? ((seg.v1 - seg.v0) / seg.dt) : 0;
          const e = Math.exp(dtp / ttn);
          const Iint = (seg.v0*ttn*(e - 1)) + (k * (ttn*(dtp - ttn)*e + ttn*ttn));
          return A * (ii0 + (Iint / Math.max(1e-18, L)));
        }
      }
    }

    return i0;
  }

  function iCAt(tm){ return C * dvSrcAt(tm); }
  function iAt(tm){ return iRLAt(tm) + iCAt(tm); }
  function vCAt(tm){ return vSrcAt(tm); }
  function vLAt(tm){
    const irl = iRLAt(tm);
    return vSrcAt(tm) - R*irl;
  }

  return {
    R,L,C,T,Ton,Toff,V,
    Tr:TrEff, Tf:TfEff, Th,
    tt, i0,
    vSrcAt, dvSrcAt,
    iRLAt, iCAt, iAt,
    vCAt, vLAt
  };
}

export function rlcPulseParallelWave({ tStart, span, N }){
  const dt = span / Math.max(1, (N - 1));
  const vSrc = new Array(N);
  const i    = new Array(N);
  const vL   = new Array(N);
  const vC   = new Array(N);

  const sol = rlcPulseParallelSteadyState();
  const T = sol.T || periodPulse();

  for(let k=0;k<N;k++){
    const t = tStart + k*dt;
    const tm = ((t % T) + T) % T;
    vSrc[k] = sol.vSrcAt(tm);
    vC[k]   = sol.vCAt(tm);
    i[k]    = sol.iAt(tm);
    vL[k]   = sol.vLAt(tm);
  }

  return { dt, vSrc, i, vL, vC, hasVC: true };
}