// rl_pulse.js
import { state } from "./state.js";
import { Rtotal, tau, periodPulse, duty } from "./core/derived.js";

export function steadyI0(){
  const T = periodPulse();
  const Ton = T*duty();
  const Toff = T - Ton;
  const tt = tau();
  if(tt <= 0) return 0;

  const Rt = Rtotal();
  const a = Math.exp(-Ton/tt);
  const b = Math.exp(-Toff/tt);
  const Iinf = state.V / Rt;

  if(Ton < 1e-9) return 0;

  const denom = (1 - a*b);
  if(Math.abs(denom) < 1e-12) return Iinf;
  return Iinf*(1-a)*b/denom;
}

export function i_on(t, I0, Iinf, tt){
  return Iinf + (I0 - Iinf)*Math.exp(-t/tt);
}

export function vL_on(t, I0, Iinf, tt){
  return Rtotal()*(Iinf - I0)*Math.exp(-t/tt);
}

export function i_off(t, Iend, tt){
  return Iend*Math.exp(-t/tt);
}

export function vL_off(t, Iend, tt){
  return -Rtotal()*Iend*Math.exp(-t/tt);
}

export function meanI2_pulse(){
  const Rt = Rtotal();
  const tt = tau();
  const T = periodPulse();
  const Ton = T*duty();
  const Toff = T - Ton;

  const Iinf = state.V / Rt;
  const I0 = steadyI0();
  const a = (I0 - Iinf);

  const e1 = Math.exp(-Ton/Math.max(1e-12,tt));
  const e2 = Math.exp(-2*Ton/Math.max(1e-12,tt));
  const int_on2 =
    (Iinf*Iinf)*Ton +
    (2*Iinf*a)*tt*(1 - e1) +
    (a*a)*(tt/2)*(1 - e2);

  const Iend = i_on(Ton, I0, Iinf, tt);

  const eoff2 = Math.exp(-2*Toff/Math.max(1e-12,tt));
  const int_off2 = (Iend*Iend)*(tt/2)*(1 - eoff2);

  return (int_on2 + int_off2) / Math.max(1e-12, T);
}

export function i_at_time_pulse(t){
  const tt = tau();
  const T = periodPulse();
  const Ton = T*duty();
  const Rt = Rtotal();
  const Iinf = state.V / Rt;
  const I0 = steadyI0();
  const Iend = i_on(Ton, I0, Iinf, tt);

  if(t <= 0) return I0;
  if(t < Ton) return i_on(t, I0, Iinf, tt);
  return i_off(t - Ton, Iend, tt);
}

export function computeTransientQuantitiesPulse(){
  const tt = tau();
  const T = periodPulse();
  const Ton = T*duty();

  const Rt = Rtotal();
  const Iinf = state.V / Rt;
  const I0 = steadyI0();

  const vL0 = Rt * (Iinf - I0);
  const didt0 = (Iinf - I0) / Math.max(1e-12, tt);

  const Iend = i_on(Ton, I0, Iinf, tt);

  const Eend = 0.5 * state.L * (Iend*Iend);
  const Toff = T - Ton;
  const Erem = Eend * Math.exp(-2*Toff/Math.max(1e-12, tt));

  const softness = Math.abs(vL0) / Math.max(1e-12, state.V);

  return { tt, T, Ton, Iinf, I0, vL0, didt0, Iend, Eend, Erem, softness };
}
