// Broadcast ticker version (clean, single-line, no duplication)

function pick(arr){
  if(!arr || !arr.length) return null;
  return arr[Math.floor(Math.random() * arr.length)];
}

function normLines(txt){
  return (txt || "")
    .split(/\r?\n/g)
    .map(s => String(s)
      .replace(/\t+/g, " ")
      .replace(/\s+/g, " ")
      .trim()
    )
    .filter(Boolean);
}

async function fetchLines(url){
  const res = await fetch(url, { cache: "no-cache" });
  if(!res.ok) throw new Error(`HTTP ${res.status}`);
  const raw = await res.text();
  return normLines(raw);
}

function escapeHtml(s){
  return String(s)
    .replaceAll("&","&amp;")
    .replaceAll("<","&lt;")
    .replaceAll(">","&gt;")
    .replaceAll('"',"&quot;")
    .replaceAll("'","&#039;");
}

function computeDuration(text){
  // Speed tuned for readability
  const base = 8;
  const perChar = 0.18 * text.length;
  return Math.max(16, Math.min(60, base + perChar));
}

export async function initQuoteLine({
  el,
  url = "/partials/marquee.txt",
  pauseBetween = 2500   // pause after line leaves screen (ms)
} = {}){
  if(!el) return;

  let lines = [];
  try{
    lines = await fetchLines(url);
  }catch(err){
    console.warn("ticker: fetch failed, using fallback");
    lines = [
      "Brain: Closed-form solutions only, Pinky.",
      "Pinky: Narf! The loop is tilting again!",
      "Brain: Adjust damping before world domination."
    ];
  }

  async function runTicker(){
    const line = pick(lines);
    if(!line) return;

    const safe = escapeHtml(line);

    el.classList.remove("scrolling");
    el.innerHTML = `<div class="tickerTrack">“${safe}”</div>`;

    const track = el.querySelector(".tickerTrack");
    if(!track) return;

    const dur = computeDuration(line);
    el.style.setProperty("--tickerDur", `${dur}s`);

    // trigger animation
    void track.offsetWidth;
    el.classList.add("scrolling");

    // wait for animation to complete
    setTimeout(() => {
      el.classList.remove("scrolling");
      setTimeout(runTicker, pauseBetween);
    }, dur * 1000);
  }

  runTicker();
}