// app/perf.js
// Small perf HUD: FPS, avg frame ms, p95, load%, long tasks, heap (Chrome-only)
// + Split timing: PHYSICS vs RENDER (per-frame accumulation)

function nowMs(){ return performance.now(); }

function percentile(arr, p){
  if(!arr.length) return 0;
  const a = Array.from(arr).sort((x,y)=>x-y);
  const idx = Math.min(a.length-1, Math.max(0, Math.floor((p/100) * (a.length-1))));
  return a[idx];
}

export function createPerfHUD({
  dotEl,
  panelEl,
  sampleWindowMs = 1000,
  targetFrameMs = 1000/60
}){
  const st = {
    on: false,
    lastTick: nowMs(),
    frames: 0,

    // frame buffers
    dtBuf: [],
    physBuf: [],
    rendBuf: [],

    // aggregates
    fps: 0,
    avgMs: 0,
    p95Ms: 0,
    load: 0,

    physAvgMs: 0,
    rendAvgMs: 0,
    physP95Ms: 0,
    rendP95Ms: 0,

    longTasks: 0,
    heapMB: null,

    // frame timing
    _frameStart: 0,
    _physThis: 0,
    _rendThis: 0,

    // section timer
    _ticT: 0,
    _ticKey: ""
  };

  // Long-task observer (main thread stalls)
  try{
    const obs = new PerformanceObserver((list)=>{
      const n = list.getEntries()?.length || 0;
      st.longTasks += n;
    });
    obs.observe({ entryTypes: ["longtask"] });
  }catch(_){}

  function updateHeap(){
    const pm = performance.memory;
    if(pm && typeof pm.usedJSHeapSize === "number"){
      st.heapMB = pm.usedJSHeapSize / (1024*1024);
    }else{
      st.heapMB = null;
    }
  }

  function dotColor(load){
    if(!isFinite(load)) return "";
    if(load < 0.60) return "rgba(120,255,160,0.95)";  // green
    if(load < 1.00) return "rgba(255,210,77,0.95)";   // yellow
    return "rgba(255,110,110,0.95)";                  // red
  }

  function renderPanel(){
    if(!panelEl) return;
    if(!st.on){ panelEl.style.display = "none"; return; }

    updateHeap();

    const heapTxt = (st.heapMB == null) ? "—" : `${st.heapMB.toFixed(1)} MB`;

    const physPct = (st.avgMs > 1e-9) ? (100 * st.physAvgMs / st.avgMs) : 0;
    const rendPct = (st.avgMs > 1e-9) ? (100 * st.rendAvgMs / st.avgMs) : 0;

    panelEl.style.display = "block";
    panelEl.innerHTML = `
      <div class="perfHead">
        <div class="perfTitle">Performance</div>
        <div class="perfSub">Frame budget @60Hz: ${targetFrameMs.toFixed(2)} ms</div>
      </div>

      <div class="perfGrid">
        <div class="perfK">FPS</div><div class="perfV">${st.fps.toFixed(1)}</div>
        <div class="perfK">Frame avg</div><div class="perfV">${st.avgMs.toFixed(2)} ms</div>
        <div class="perfK">Frame p95</div><div class="perfV">${st.p95Ms.toFixed(2)} ms</div>
        <div class="perfK">Load</div><div class="perfV">${(st.load*100).toFixed(0)}%</div>

        <div class="perfK">Physics avg</div><div class="perfV">${st.physAvgMs.toFixed(2)} ms (${physPct.toFixed(0)}%)</div>
        <div class="perfK">Render avg</div><div class="perfV">${st.rendAvgMs.toFixed(2)} ms (${rendPct.toFixed(0)}%)</div>
        <div class="perfK">Physics p95</div><div class="perfV">${st.physP95Ms.toFixed(2)} ms</div>
        <div class="perfK">Render p95</div><div class="perfV">${st.rendP95Ms.toFixed(2)} ms</div>

        <div class="perfK">Long tasks</div><div class="perfV">${st.longTasks}</div>
        <div class="perfK">JS heap</div><div class="perfV">${heapTxt}</div>
      </div>

      <div class="perfHint">
        Dot: green &lt;60%, yellow 60–100%, red &gt;100% (can’t hold 60fps).
      </div>
    `;
  }

  function aggregate(){
    const t = nowMs();
    const elapsed = t - st.lastTick;
    if(elapsed < sampleWindowMs) return;

    st.fps = (st.frames * 1000) / Math.max(1e-9, elapsed);

    const avg = st.dtBuf.length ? st.dtBuf.reduce((a,b)=>a+b,0) / st.dtBuf.length : 0;
    const physAvg = st.physBuf.length ? st.physBuf.reduce((a,b)=>a+b,0) / st.physBuf.length : 0;
    const rendAvg = st.rendBuf.length ? st.rendBuf.reduce((a,b)=>a+b,0) / st.rendBuf.length : 0;

    st.avgMs = avg;
    st.p95Ms = percentile(st.dtBuf, 95);
    st.load  = avg / targetFrameMs;

    st.physAvgMs = physAvg;
    st.rendAvgMs = rendAvg;
    st.physP95Ms = percentile(st.physBuf, 95);
    st.rendP95Ms = percentile(st.rendBuf, 95);

    st.frames = 0;
    st.dtBuf.length = 0;
    st.physBuf.length = 0;
    st.rendBuf.length = 0;
    st.lastTick = t;

    if(dotEl){
      dotEl.style.background = dotColor(st.load) || dotEl.style.background;
      dotEl.title =
        `Perf: ${st.fps.toFixed(1)} fps, avg ${st.avgMs.toFixed(2)} ms, `
        + `phys ${st.physAvgMs.toFixed(2)} ms, rend ${st.rendAvgMs.toFixed(2)} ms`;
    }

    renderPanel();
  }

  function beginFrame(){
    st._frameStart = nowMs();
    st._physThis = 0;
    st._rendThis = 0;
    st._ticKey = "";
    st._ticT = 0;
  }

  function tic(key){
    st._ticKey = key || "";
    st._ticT = nowMs();
  }

  function toc(){
    const dt = nowMs() - (st._ticT || nowMs());
    const k = st._ticKey;

    if(k === "physics") st._physThis += dt;
    else if(k === "render") st._rendThis += dt;
    else st._rendThis += dt; // default bucket (safe)

    st._ticKey = "";
    st._ticT = 0;
  }

  function endFrame(){
    const dt = nowMs() - st._frameStart;

    st.dtBuf.push(dt);
    st.physBuf.push(st._physThis);
    st.rendBuf.push(st._rendThis);

    st.frames++;
    aggregate();
  }

  function setOn(v){
    st.on = !!v;
    renderPanel();
  }
  function toggle(){ setOn(!st.on); }

  if(dotEl){
    dotEl.style.cursor = "pointer";

    dotEl.setAttribute("role", "button");
    dotEl.setAttribute("aria-label", "Toggle performance panel");
    dotEl.tabIndex = 0;

    dotEl.addEventListener("click", toggle);
    dotEl.addEventListener("keydown", (e) => {
      if(e.key === "Enter" || e.key === " "){
        e.preventDefault();
        toggle();
      }
    });
  }

  // note: tic/toc are optional — if you never call them, physics/render stays 0/0
  return { beginFrame, endFrame, tic, toc, setOn, toggle, state: st };
}