// panels/power_panel.js
import { $, fmt, fmtJ } from "../dom.js";
import { state } from "../state.js";
import {
  powerSplit,
  powerQualityMetrics,
  windowCapacitorMetrics,
  windowMagneticMetrics,
  energyFlowMetrics,
  rlcBreakdownFundamental
} from "../metrics.js";

import { updatePeriodicityBadge } from "./periodicity_badge.js";
import { drawPhasorPlot } from "./phasor_plot.js";

export function updatePowerPanel(){
  const { Psrc, Pload, Pcoil } = powerSplit();

  const pq  = powerQualityMetrics();
  const cap = windowCapacitorMetrics();
  const wm  = windowMagneticMetrics();

  // === Fundamental phasor view (PQ tab) ===
  drawPhasorPlot(pq);

  // =======================
  // RLC / Resonance / Q split
  // =======================
  const rlc = rlcBreakdownFundamental(pq?.Irms, pq?.Vrms);

  if(rlc){
    if($('#XLohm'))   $('#XLohm').textContent   = `${fmt(rlc.XL)} Ω`;
    if($('#Xcohm'))   $('#Xcohm').textContent   = `${fmt(rlc.Xc)} Ω`;
    if($('#Xnetohm')) $('#Xnetohm').textContent = `${fmt(rlc.X)} Ω`;

    if($('#f0'))      $('#f0').textContent      = `${rlc.f0.toFixed(2)} Hz`;
    if($('#detune'))  $('#detune').textContent  = `${rlc.detune.toExponential(2)}`;
    if($('#Q0'))      $('#Q0').textContent      = `${rlc.Q0.toFixed(3)}`;
    if($('#BW'))      $('#BW').textContent      = `${rlc.BW.toFixed(2)} Hz`;

    // Energy tab (Q split)
    if($('#QLvar'))   $('#QLvar').textContent   = `${fmt(rlc.QL)} var`;
    if($('#QCvar'))   $('#QCvar').textContent   = `${fmt(rlc.QC)} var`;
    if($('#Qnetvar')) $('#Qnetvar').textContent = `${fmt(rlc.Qnet)} var`;

    // PQ tab (Q split)
    if($('#QLvarPQ'))   $('#QLvarPQ').textContent   = `${fmt(rlc.QL)} var`;
    if($('#QCvarPQ'))   $('#QCvarPQ').textContent   = `${fmt(rlc.QC)} var`;
    if($('#QnetvarPQ')) $('#QnetvarPQ').textContent = `${fmt(rlc.Qnet)} var`;
  }else{
    [
      'XLohm','Xcohm','Xnetohm',
      'f0','detune','Q0','BW',
      'QLvar','QCvar','Qnetvar',
      'QLvarPQ','QCvarPQ','QnetvarPQ'
    ].forEach(id=>{
      const el = document.getElementById(id);
      if(el) el.textContent = '—';
    });
  }

  // =======================
  // Power Quality
  // =======================
  if($('#Vrms'))   $('#Vrms').textContent   = `${fmt(pq.Vrms)} V`;
  if($('#Irms'))   $('#Irms').textContent   = `${fmt(pq.Irms)} A`;
  if($('#Sva'))    $('#Sva').textContent    = `${fmt(pq.S)} VA`;
  if($('#PF'))     $('#PF').textContent     = `${fmt(pq.PF)}`;
  if($('#phiDeg')) $('#phiDeg').textContent = `${(pq.phiDeg ?? 0).toFixed(2)}°`;
  if($('#Qvar'))   $('#Qvar').textContent   = `${fmt(pq.Q)} var`;

  // =======================
  // Magnetic window
  // =======================
  if($('#dIwinLabel'))
    $('#dIwinLabel').textContent = `Δi_win (0→τ): ${fmt(wm.dIwin)} A`;

  if($('#Qwin'))    $('#Qwin').textContent    = `${fmt(wm.Qwin)} C`;
  if($('#dLamWin')) $('#dLamWin').textContent = `${fmt(wm.dLamWin)} V·s`;

  document.querySelectorAll('#WLpkWin').forEach(el=>{
    el.textContent = fmtJ(wm.WLpkWin);
  });

  // =======================
  // Capacitor window
  // =======================
  if(!cap){
    ['VCrms','VCpkWin','dQcapWin','dWcapWin','chkWC'].forEach(id=>{
      const el = document.getElementById(id);
      if(el) el.textContent = '—';
    });
    document.querySelectorAll('#WCpkWin').forEach(el=>{
      el.textContent = '—';
    });
  }else{
    if($('#VCrms'))    $('#VCrms').textContent    = `${fmt(cap.VCrmsWin)} V`;
    if($('#VCpkWin'))  $('#VCpkWin').textContent  = `${fmt(cap.pkAbsVC)} V`;
    if($('#dQcapWin')) $('#dQcapWin').textContent = `${fmt(cap.dQcapWin)} C`;
    if($('#dWcapWin')) $('#dWcapWin').textContent = fmtJ(cap.dWcapWin);

    document.querySelectorAll('#WCpkWin').forEach(el=>{
      el.textContent = fmtJ(cap.WCpkWin);
    });

    if($('#chkWC')){
      $('#chkWC').textContent =
        `ΔWc − ∫vC·i dt = ${fmtJ(cap.diffW)}  (${(cap.relW*100).toFixed(6)}%)`;
    }
  }

  // =======================
  // Checks
  // =======================
  if($('#chkLam')){
    const lhs = wm.dLamWin;
    const rhs = state.L * wm.dIwin;
    const diff = lhs - rhs;
    const rel = Math.abs(diff) / Math.max(1e-12, Math.abs(lhs), Math.abs(rhs));
    $('#chkLam').textContent =
      `Δλ - L·Δi = ${fmt(diff)} V·s  (${(rel*100).toFixed(6)}%)`;
  }

  if($('#chkW')){
    const dW = 0.5 * state.L * (wm.iT*wm.iT - wm.i0*wm.i0);
    const rhsW = wm.ELwin;
    const diffW = dW - rhsW;
    const relW = Math.abs(diffW) / Math.max(1e-12, Math.abs(dW), Math.abs(rhsW));
    $('#chkW').textContent =
      `ΔW - ∫vL·i dt = ${fmtJ(diffW)}  (${(relW*100).toFixed(6)}%)`;
  }

  // =======================
  // Power numbers
  // =======================
  if($('#Psrc'))  $('#Psrc').textContent  = `${fmt(Psrc)} W`;
  if($('#Pload')) $('#Pload').textContent = `${fmt(Pload)} W`;
  if($('#Pcoil')) $('#Pcoil').textContent = `${fmt(Pcoil)} W`;

  // =======================
  // Energy per period
  // =======================
  const em = energyFlowMetrics();
  if($('#EgenPer'))  $('#EgenPer').textContent  = fmtJ(em.Egen_per);
  if($('#EloadPer')) $('#EloadPer').textContent = fmtJ(em.Eload_per);
  if($('#EcoilPer')) $('#EcoilPer').textContent = fmtJ(em.Ecoil_per);
  if($('#dWLtau'))   $('#dWLtau').textContent   = fmtJ(em.dWL_tau);
  if($('#balErr'))   $('#balErr').textContent   = `${(em.balanceErr*100).toFixed(3)}%`;

  // =======================
  // Share
  // =======================
  const denom = Math.max(1e-12, Math.abs(Psrc), Math.abs(Pload + Pcoil));
  if($('#shareLoad')) $('#shareLoad').textContent = `${((Pload/denom)*100).toFixed(1)}%`;
  if($('#shareCoil')) $('#shareCoil').textContent = `${((Pcoil/denom)*100).toFixed(1)}%`;

  // =======================
  // Δ vs Ref
  // =======================
  if(!state.refSnapshot || !isFinite(state.refSnapshot.Psrc)){
    if($('#dPsrc')) $('#dPsrc').textContent = '—';
    if($('#pCallout')) $('#pCallout').textContent = 'Press Set Ref to compare.';
    updatePeriodicityBadge();
    return;
  }

  const Pref = state.refSnapshot.Psrc;
  const delta = (Psrc/Pref - 1)*100;
  if($('#dPsrc')) $('#dPsrc').textContent =
    `${delta>0?'+':''}${delta.toFixed(1)}%`;

  if($('#pCallout')){
    if(Pcoil > Pload)
      $('#pCallout').textContent =
        'Loss-dominated: more power dissipates in the coil than in the load.';
    else if(Pload > Pcoil)
      $('#pCallout').textContent =
        'Load-dominated: more power reaches the load than coil loss.';
    else
      $('#pCallout').textContent =
        'Balanced split between load and coil loss.';
  }

  updatePeriodicityBadge();
}