// panels/phasor_plot.js
import { state } from "../state.js";

export function drawPhasorPlot(pq){
  const canvas = document.getElementById("phasorPlot");
  if(!canvas || !pq) return;

  const ctx = canvas.getContext("2d");
  const dpr = window.devicePixelRatio || 1;

  const rect = canvas.getBoundingClientRect();
  canvas.width  = rect.width  * dpr;
  canvas.height = rect.height * dpr;
  ctx.setTransform(dpr, 0, 0, dpr, 0, 0);

  const w = rect.width;
  const h = rect.height;
  ctx.clearRect(0,0,w,h);

  // Theme colors from tokens
  const css = getComputedStyle(document.documentElement);
  const colV    = css.getPropertyValue('--e').trim()    || 'rgba(77,227,255,0.95)'; // CH1
  const colI    = css.getPropertyValue('--p').trim()    || 'rgba(255,210,77,0.92)'; // CH2
  const colTxt  = css.getPropertyValue('--text').trim() || '#e6edf3';
  const colGrid = css.getPropertyValue('--muted').trim()|| 'rgba(151,166,178,.85)';

  const cx = w/2;
  const cy = h/2;
  const rMax = Math.min(w,h)*0.38;

  const Vr = pq.Vrms || 0;
  const Ir = pq.Irms || 0;
  const phi = (pq.phiDeg || 0) * Math.PI/180;

  const magMax = Math.max(Vr, Ir, 1e-9);
  const vLen = rMax * (Vr/magMax);
  const iLen = rMax * (Ir/magMax);

  // grid circle
  ctx.strokeStyle = colGrid;
  ctx.lineWidth = 1;
  ctx.beginPath();
  ctx.arc(cx, cy, rMax, 0, 2*Math.PI);
  ctx.stroke();

  // axes
  ctx.strokeStyle = colTxt;
  ctx.beginPath();
  ctx.moveTo(cx-rMax, cy);
  ctx.lineTo(cx+rMax, cy);
  ctx.moveTo(cx, cy-rMax);
  ctx.lineTo(cx, cy+rMax);
  ctx.stroke();

  function arrow(x0,y0,x1,y1,color,label){
    ctx.strokeStyle = color;
    ctx.fillStyle   = color;
    ctx.lineWidth = 2;

    ctx.beginPath();
    ctx.moveTo(x0,y0);
    ctx.lineTo(x1,y1);
    ctx.stroke();

    const ang = Math.atan2(y1-y0, x1-x0);
    const head = 8;

    ctx.beginPath();
    ctx.moveTo(x1,y1);
    ctx.lineTo(x1 - head*Math.cos(ang - Math.PI/6),
               y1 - head*Math.sin(ang - Math.PI/6));
    ctx.lineTo(x1 - head*Math.cos(ang + Math.PI/6),
               y1 - head*Math.sin(ang + Math.PI/6));
    ctx.closePath();
    ctx.fill();

    ctx.font = "12px system-ui";
    ctx.fillText(label, x1 + 6, y1 - 6);
  }

  // V reference
  arrow(cx, cy, cx + vLen, cy, colV, "V₁");

  // I at -phi (lagging if inductive)
  const angI = -phi;
  arrow(
    cx, cy,
    cx + iLen*Math.cos(angI),
    cy + iLen*Math.sin(angI),
    colI,
    "I₁"
  );

  // angle arc
  ctx.strokeStyle = colI;
  ctx.lineWidth = 1.5;
  ctx.beginPath();
  ctx.arc(cx, cy, rMax*0.35, 0, angI, angI<0);
  ctx.stroke();

  // info text
  ctx.fillStyle = colTxt;
  ctx.font = "11px system-ui";
  ctx.fillText(
    `${state.srcMode === 'pulse' ? 'Pulse (fundamental)' : 'Sine'} · φ=${pq.phiDeg?.toFixed(2) ?? 0}°`,
    12, 16
  );
}