// panels/meas_strip.js
import { $ } from "../dom.js";
import { state } from "../state.js";
import { tau } from "../core/derived.js";
import { powerQualityMetrics, rlcBreakdownFundamental } from "../metrics.js";

export function updateMeasStrip(){
  // ---- helpers ----
  const elModel = $('#measModel');
  const elExc   = $('#measExc');
  const elWin   = $('#measWin');
  const elAss   = $('#measAss'); // repurposed line as "Resonance"

  function setLineValue(el, txt){
    if(el) el.textContent = txt;
  }

  // rename the left key label ("Assumptions" -> "Resonance") without touching HTML
  function renameKeyForValueEl(valueEl, newKey){
    const line = valueEl?.closest?.('.measLine');
    const key  = line?.querySelector?.('.measK');
    if(key) key.textContent = newKey;
  }

  function fmtDiv(ms){
    if(!isFinite(ms) || ms <= 0) return '—';
    if(ms < 1) return `${(ms*1000).toFixed(2)} µs/div`;
    if(ms < 10) return `${ms.toFixed(3)} ms/div`;
    return `${ms.toFixed(2)} ms/div`;
  }

  function fmtMs(s){
    if(!isFinite(s)) return '—';
    return `${(s*1e3).toFixed(3)} ms`;
  }

  // ---- Model ----
  setLineValue(elModel, 'Analytic single-branch series RL/RLC');

  // ---- Excitation (instrument-style) ----
  const tb = fmtDiv(state.timeDiv_ms);
  let exc = '—';

  if(state.srcMode === 'sine'){
    const Vpk = Math.max(0, state.V || 0);
    const f   = Math.max(0, state.fs || 0);

    const pq = powerQualityMetrics?.() || null;
    const phi = (pq && isFinite(pq.phiDeg)) ? pq.phiDeg : null;

    exc =
      `Sine · ${Vpk.toFixed(0)} Vpk · ${f.toFixed(0)} Hz` +
      (phi == null ? '' : ` · φ=${phi.toFixed(1)}°`) +
      ` · ${tb}`;
  }else{
    const V = Math.max(0, state.V || 0);
    const f = Math.max(0, state.f || 0);
    const d = Math.max(0, state.dutyPct || 0);

    exc =
      `Pulse · ${V.toFixed(0)} V · ${f.toFixed(0)} Hz · ${d.toFixed(2)}% · ${tb}`;
  }

  setLineValue(elExc, exc);

  // ---- Window ----
  const tt = tau?.() ?? NaN;
  const span = (state.timeDiv_ms*1e-3) * 10; // 10 divisions
  const win =
    `0 → τ (τ=${fmtMs(tt)}) · span=${fmtMs(span)}`;

  setLineValue(elWin, win);

  // ---- Resonance (derived results, not “assumptions”) ----
  const pq = powerQualityMetrics?.() || null;
  const rlc = rlcBreakdownFundamental?.(pq?.Irms, pq?.Vrms) || null;

  let res = '—';

  if(rlc && isFinite(rlc.f0)){
    const f0 = rlc.f0;
    const Q0 = rlc.Q0;
    const BW = rlc.BW;
    const det = rlc.detune;

    res =
      `f₀=${f0.toFixed(2)} Hz · ` +
      `Q₀=${isFinite(Q0) ? Q0.toFixed(3) : '—'} · ` +
      `BW=${isFinite(BW) ? BW.toFixed(2) : '—'} Hz · ` +
      `detune=${isFinite(det) ? det.toExponential(2) : '—'}`;

    if(rlc.topology === 'C_parallel_RL' && isFinite(rlc.Zpeak)){
      res += ` · Z_peak=${rlc.Zpeak.toFixed(2)} Ω`;
    }
  }

  setLineValue(elAss, res);
  renameKeyForValueEl(elAss, 'Resonance');
}