// panels/derived_panel.js
import { $, fmt, fmtJ } from "../dom.js";
import { state } from "../state.js";
import { Rtotal, tau } from "../core/derived.js";
import { sineParams } from "../rl_sine.js";
import { computeTransientQuantitiesPulse } from "../rl_pulse.js";
import {
  sineParamsAny,
  powerQualityMetrics,
  windowCapacitorMetrics,
  rlcBreakdownSine
} from "../metrics.js";

export function updateDerived(){
  const Rt = Rtotal();           // RL-branch resistance (Rcoil+Rload)
  const tt = tau();              // RL-branch time constant L/Rt

  const rEl = $('#Rtotal');
  if(rEl) rEl.textContent = `${fmt(Rt)} Ω`;

  const tauEl = $('#tau');
  if(tauEl) tauEl.textContent = `${fmt(tt*1e3)} ms`;

  const lblIinf = $('#lblIinf');
  const lblI0   = $('#lblI0');

  const topo = (state.topology || "seriesRLC");
  const topoNorm = (topo === "parallelRLC") ? "C_parallel_RL" : topo;
  const C = Math.max(0, state.C ?? 0);
  const isParallelC = (topoNorm === "C_parallel_RL" && C > 0);

  // Topology-aware labels for the values under Rload
  const rLbl = $('#RtotalLabel');
  if(rLbl) rLbl.textContent = isParallelC ? 'R_RL = R_coil + R_load' : 'R_total = R_coil + R_load';

  const tLbl = $('#tauLabel');
  if(tLbl) tLbl.textContent = isParallelC ? 'τ_RL = L / R_RL' : 'τ = L / R_total';

  const ZinEl = $('#Zin');
  const GinEl = $('#Gin');
  const BinEl = $('#Bin');

  if(state.srcMode === 'sine' && isParallelC){
    const sp = sineParamsAny();

    if(sp && isFinite(sp.G) && isFinite(sp.B) && isFinite(sp.Irms) && isFinite(sp.Vrms)){
      const Zin = (sp.Irms > 1e-18) ? (sp.Vrms / sp.Irms) : 0;

      if(ZinEl) ZinEl.textContent = `${fmt(Zin)} Ω`;
      if(GinEl) GinEl.textContent = `${fmt(sp.G)} S`;
      if(BinEl) BinEl.textContent = `${fmt(sp.B)} S`;
    }else{
      if(ZinEl) ZinEl.textContent = '—';
      if(GinEl) GinEl.textContent = '—';
      if(BinEl) BinEl.textContent = '—';
    }
  }else{
    if(ZinEl) ZinEl.textContent = '—';
    if(GinEl) GinEl.textContent = '—';
    if(BinEl) BinEl.textContent = '—';
  }

  // -------------------------
  // SINE MODE (topology-aware PQ layer)
  // -------------------------
  if(state.srcMode === 'sine'){
    const pq = powerQualityMetrics();

    if(lblIinf) lblIinf.textContent = isParallelC ? 'I_rms (total, sine)' : 'I_rms (sine)';
    if(lblI0)   lblI0.textContent   = isParallelC ? 'Phase φ (V ↔ I_total)' : 'Phase lag φ';

    if($('#iinf')) $('#iinf').textContent = (pq && isFinite(pq.Irms)) ? `${fmt(pq.Irms)} A` : '—';
    if($('#i0'))   $('#i0').textContent   = (pq && isFinite(pq.phiDeg)) ? `${pq.phiDeg.toFixed(2)}°` : '—';

    // Peak vL + peak di/dt (series only)
    if(isParallelC){
      if($('#pkVL'))     $('#pkVL').textContent     = '—';
      if($('#pkDIDT'))   $('#pkDIDT').textContent   = '—';
      if($('#softIdx'))  $('#softIdx').textContent  = '—';
    }else{
      const sp = sineParams();
      if(!sp || !isFinite(sp.Ipk) || !isFinite(sp.w) || !isFinite(sp.phi)){
        if($('#pkVL'))    $('#pkVL').textContent    = '—';
        if($('#pkDIDT'))  $('#pkDIDT').textContent  = '—';
        if($('#softIdx')) $('#softIdx').textContent = '—';
        return;
      }
      const N = 800;
      const dt = (tt > 0) ? (tt/Math.max(1, N)) : (1/(Math.max(1e-18, state.fs)*N));
      let pkVL = 0;
      let pkDidt = 0;
      for(let k=0;k<N;k++){
        const t = (k+0.5)*dt;
        const didt = sp.w * sp.Ipk * Math.cos(sp.w*t - sp.phi);
        const vL = state.L * didt;
        pkVL = Math.max(pkVL, Math.abs(vL));
        pkDidt = Math.max(pkDidt, Math.abs(didt));
      }
      const soft = pkVL / Math.max(1e-12, state.V);

      if($('#pkVL'))    $('#pkVL').textContent    = `${fmt(pkVL)} V`;
      if($('#pkDIDT'))  $('#pkDIDT').textContent  = `${fmt(pkDidt)} A/s`;
      if($('#softIdx')) $('#softIdx').textContent = `${(soft*100).toFixed(1)}% of V`;
    }

    // Capacitor stress (topology-aware)
    const capM = windowCapacitorMetrics();
    const pkVCEl = $('#pkVC');
    const pkDVCDTEl = $('#pkDVCDT');

    if(capM && C > 0){
      if(pkVCEl) pkVCEl.textContent = `${fmt(capM.pkAbsVC)} V`;

      if(pkDVCDTEl){
        if(isParallelC){
          const w = 2*Math.PI*Math.max(1e-18, state.fs);
          const pkDvdt = Math.abs(w * state.V);
          pkDVCDTEl.textContent = `${fmt(pkDvdt)} V/s`;
        }else{
          const rlcS = rlcBreakdownSine();
          const iPk = rlcS ? rlcS.Ipk : null;
          if(iPk != null){
            const pkDvdt = Math.abs(iPk) / Math.max(1e-30, C);
            pkDVCDTEl.textContent = `${fmt(pkDvdt)} V/s`;
          }else{
            pkDVCDTEl.textContent = '—';
          }
        }
      }
    }else{
      if(pkVCEl) pkVCEl.textContent = '—';
      if(pkDVCDTEl) pkDVCDTEl.textContent = '—';
    }

    if($('#Iend')) $('#Iend').textContent = '—';
    if($('#Eend')) $('#Eend').textContent = '—';
    if($('#Erem')) $('#Erem').textContent = '—';
    return;
  }

  // -------------------------
  // PULSE MODE (series branch)
  // -------------------------
  if(lblIinf) lblIinf.textContent = 'I∞ = V/R_total';
  if(lblI0)   lblI0.textContent   = 'I0 (steady)';

  const q = computeTransientQuantitiesPulse();

  if($('#iinf')) $('#iinf').textContent = `${fmt(q.Iinf)} A`;
  if($('#i0'))   $('#i0').textContent   = `${fmt(q.I0)} A`;

  if($('#pkVL'))    $('#pkVL').textContent    = `${fmt(Math.abs(q.vL0))} V`;
  if($('#pkDIDT'))  $('#pkDIDT').textContent  = `${fmt(Math.abs(q.didt0))} A/s`;
  if($('#softIdx')) $('#softIdx').textContent = `${(q.softness*100).toFixed(1)}% of V`;

  if($('#Iend')) $('#Iend').textContent = `${fmt(q.Iend)} A`;
  if($('#Eend')) $('#Eend').textContent = fmtJ(q.Eend);
  if($('#Erem')) $('#Erem').textContent = `${fmtJ(q.Erem)} (after OFF)`;
}