// app/metrics/status_line.js
// Purpose: cheap one-line diagnostic/status text for the meas strip / HUD.
// NOTE: Must NOT import from ../metrics.js to avoid circular deps.

import { state } from "../state.js";
import { Rtotal } from "../core/derived.js";

import { powerQualityMetrics } from "./pq.js";
import { rlcBreakdownFundamental } from "./rlc_breakdown.js";
import { harmonicMetricsPulse } from "./harmonics.js";

export function resonanceHarmonicsLine(){
  // One-line “smart” note suitable for meas strip / status areas.
  // IMPORTANT: Must be cheap (called frequently) → harmonic part is throttled.

  const Rt = Rtotal();

  const pq = powerQualityMetrics();
  const rlc = rlcBreakdownFundamental(pq?.Irms, pq?.Vrms);

  let res = "";
  if(rlc){
    const fDrive = (state.srcMode === "sine") ? (state.fs || 0) : (state.f || 0);
    const det = rlc.detune;
    const near = Math.abs(det) < 0.02; // ±2% of ω0

    res =
      `Res: f0=${rlc.f0.toFixed(2)} Hz, f=${(fDrive||0).toFixed(2)} Hz, ` +
      `Δ=${det.toExponential(2)}, Q0=${rlc.Q0.toFixed(2)}, BW=${rlc.BW.toFixed(2)} Hz` +
      (near ? " (near)" : "");
  }else if((state.C||0) <= 0){
    res = "Res: C=0 (RL mode)";
  }else{
    res = "Res: —";
  }

  let harm = "";
  if(state.srcMode === "sine"){
    harm = "Harm: ideal sine";
  }else{
    // throttled internally (default 250ms)
    const hm = harmonicMetricsPulse({ Nh: 7, N: 3000, throttle_ms: 250 });
    if(hm){
      const thdi = (hm.THDi*100).toFixed(2);
      const thdv = (hm.THDv*100).toFixed(2);
      harm = `Harm: THD(i)=${thdi}% THD(v)=${thdv}%`;
    }else{
      harm = "Harm: —";
    }
  }

  return `${res} · ${harm} · R_total=${Rt.toFixed(2)} Ω`;
}