// app/metrics/rlc_breakdown.js
// Purpose: topology-aware RLC reactance / resonance breakdown (reference-grade).
// NOTE: Must NOT import from ../metrics.js to avoid circular deps.

import { state } from "../state.js";
import { Rtotal } from "../core/derived.js";
import { rlcSineParamsSeries } from "../rlc_sine.js";

export function rlcBreakdownSine(){
  // Series RLC (steady-state sine) breakdown
  const C = Math.max(0, state.C ?? 0);
  if(state.srcMode !== "sine") return null;
  if((state.topology || "seriesRLC") !== "seriesRLC") return null;
  if(!(C > 0)) return null;

  const sp = rlcSineParamsSeries(); // provides: w, R, Xc maybe, etc.
  const R = sp.R;
  const w = sp.w;

  const XL = w * state.L;                 // ωL
  const Xc = sp.Xc ?? (1/(w*C));          // fallback if kernel doesn't export Xc yet
  const X  = XL - Xc;

  // Resonance
  const w0 = 1 / Math.sqrt(Math.max(1e-30, state.L * C));
  const f0 = w0 / (2*Math.PI);

  // Detuning (dimensionless)
  const detune = (w - w0) / Math.max(1e-30, w0);

  // Quality factor at resonance (series RLC)
  // Q0 = ω0 L / R = 1/(ω0 C R)
  const Q0 = (w0 * state.L) / Math.max(1e-30, R);

  // Bandwidth (approx, small damping): BW = f0 / Q0
  const BW = f0 / Math.max(1e-30, Q0);

  // Reactive powers split (physical, sine):
  // QL = I_rms^2 * XL (positive), QC = - I_rms^2 * Xc (negative), Qnet = I_rms^2 * (XL - Xc)
  const I2 = sp.Irms * sp.Irms;
  const QL = I2 * XL;
  const QC = -I2 * Xc;
  const Qnet = I2 * X;

  return { ...sp, XL, Xc, X, f0, Q0, BW, detune, QL, QC, Qnet };
}

// Fundamental-frequency RLC breakdown (works for BOTH seriesRLC and parallelRLC).
// For pulse mode this uses the repetition frequency as the fundamental.
// Pass IrmsFund = fundamental Irms (sine: Irms, pulse: Irms1).
// Pass VrmsFund = fundamental Vrms (sine: Vrms, pulse: Vrms1).
export function rlcBreakdownFundamental(IrmsFund, VrmsFund){
  const C = Math.max(0, state.C ?? 0);
  const topo = (state.topology || "seriesRLC");
  if(!(C > 0)) return null;

  const topoNorm = (topo === "parallelRLC") ? "C_parallel_RL" : topo;
  if(topoNorm !== "seriesRLC" && topoNorm !== "C_parallel_RL") return null;

  const f = (state.srcMode === "sine") ? (state.fs || 0) : (state.f || 0);
  const w = 2*Math.PI*Math.max(1e-12, f);

  const R = Rtotal();
  const L = Math.max(0, state.L);

  // Always expose these "component reactances" (useful in both topologies)
  const XL = w * L;
  const Xc = 1 / Math.max(1e-30, w * C);

  // Resonant frequency:
  // - series:   w0 = 1/sqrt(LC)
  // - parallel (R in series with L branch, || C): Im(Y)=0 => w0^2 = 1/(LC) - (R/L)^2
  const w0_series = 1 / Math.sqrt(Math.max(1e-30, L * C));
  const w0_parallel_sq = (1 / Math.max(1e-30, L * C)) - (R*R) / Math.max(1e-30, L*L);
  const w0 = (topoNorm === "C_parallel_RL")
    ? Math.sqrt(Math.max(0, w0_parallel_sq))
    : w0_series;

  if(!(w0 > 0)) return null;

  const f0 = w0 / (2*Math.PI);
  const detune = (w - w0) / Math.max(1e-30, w0);

  // Fundamental RMS values (needed for var split; prefer passed-in, fallback to source RMS)
  const Irms = Math.max(0, IrmsFund || 0);
  const Vrms = Math.max(
    0,
    VrmsFund || (state.V/Math.SQRT2) // safe fallback (exact for sine; approximate for pulse fundamental)
  );

  if(topoNorm === "seriesRLC"){
    const X  = XL - Xc;

    const Q0 = (w0 * L) / Math.max(1e-30, R);
    const BW = f0 / Math.max(1e-30, Q0);

    // var split (series): uses line current
    const I2 = Irms * Irms;
    const QL = I2 * XL;       // +
    const QC = -I2 * Xc;      // -
    const Qnet = I2 * X;      // +/-

    return { topology: topoNorm, XL, Xc, X, f0, detune, Q0, BW, QL, QC, Qnet, w, R };
  }

  // ---- C_parallel_RL ( (R + jωL) || (1/jωC) ) ----
  // Admittances:
  // Y_RL = 1/(R + jωL) = (R - jωL)/(R^2 + (ωL)^2)
  // Y_C  = jωC
  const denom = (R*R + XL*XL);

  const G_RL = (denom > 0) ? (R / denom) : 0;      // conductance
  const B_RL = (denom > 0) ? (-XL / denom) : 0;    // susceptance (negative)
  const B_C  = w * C;                               // positive

  const G = G_RL;
  const B = B_RL + B_C;

  const Ymag = Math.hypot(G, B);
  const Zeq = (Ymag > 0) ? (1 / Ymag) : Infinity;

  // At parallel resonance, B=0 and Zeq peaks:
  // Zeq_peak = L/(R*C)
  const Zpeak = (R > 0 && C > 0) ? (L / (R * C)) : Infinity;

  // Parallel Q0 (using Rp = Zpeak): Q0 = Rp * sqrt(C/L) = sqrt(L/C)/R
  const Q0 = (R > 0) ? (Math.sqrt(Math.max(0, L / Math.max(1e-30, C))) / R) : Infinity;
  const BW = f0 / Math.max(1e-30, Q0);

  // var split (parallel): uses Vrms and branch susceptances
  // S = V^2 * conj(Y) => Q = -V^2 * B
  const V2 = Vrms * Vrms;
  const QL = -V2 * B_RL;      // positive
  const QC = -V2 * B_C;       // negative
  const Qnet = QL + QC;

  // Provide "X" as 0 in parallel mode (avoid misleading XL-Xc usage)
  const X = 0;

  return {
    topology: topoNorm,
    XL, Xc, X,
    f0, detune, Q0, BW,
    QL, QC, Qnet,
    Zeq, Zpeak,
    G, B, w, R
  };
}