// app/metrics/pulse_solvers.js
// Purpose: cached pulse steady-state solvers (series + parallel)

import { state } from "../state.js";
import { duty } from "../core/derived.js";
import { rlcPulseSteadyState, rlcPulseParallelSteadyState } from "../rlc_pulse.js";

// ---------- Pulse SERIES RLC helpers (cached) ----------
let _pulseRlcCache = { key: "", sol: null };

export function pulseRlcSol(){
  const C = Math.max(0, state.C || 0);
  const topology = state.topology || "seriesRLC";
  if(state.srcMode !== "pulse") return null;
  if(topology !== "seriesRLC") return null;
  if(!(C > 0)) return null;

  const key = [
    state.V, state.f, state.dutyPct,
    state.L, state.Rcoil, state.Rload,
    state.C, topology
  ].join("|");

  if(_pulseRlcCache.key === key && _pulseRlcCache.sol) return _pulseRlcCache.sol;

  const sol = rlcPulseSteadyState();
  _pulseRlcCache = { key, sol };
  return sol;
}

// ---------- Pulse PARALLEL (R+L)||C helpers (cached) ----------
let _pulseParCache = { key: "", sol: null };

export function pulseRlcParallelSol(){
  const C = Math.max(0, state.C || 0);
  const topology = state.topology || "seriesRLC";
  if(state.srcMode !== "pulse") return null;
  if(topology !== "C_parallel_RL") return null;
  if(!(C > 0)) return null;

  const key = [
    state.V, state.f, state.dutyPct,
    state.L, state.Rcoil, state.Rload,
    state.C, topology
  ].join("|");

  if(_pulseParCache.key === key && _pulseParCache.sol) return _pulseParCache.sol;

  const sol = rlcPulseParallelSteadyState();
  _pulseParCache = { key, sol };
  return sol;
}

// returns seriesRLC sol or parallel sol depending on topology
export function pulseRlcSolAny(){
  return pulseRlcSol() || pulseRlcParallelSol();
}

// tiny helper (kept here because harmonics wants Ton too)
export function pulseTonFromSol(sol){
  if(!sol || !(sol.T > 0)) return 0;
  return (sol.Ton != null) ? sol.Ton : (sol.T * duty());
}