// app/metrics/pq.js
// Purpose: Power Quality metrics (Vrms/Irms/P/S/PF/Q/phi) for sine + pulse.
// NOTE: No imports from ../metrics.js to avoid circular deps.

import { state } from "../state.js";
import { duty, periodPulse } from "../core/derived.js";
import { sineParams } from "../rl_sine.js";
import { rlcSineParamsSeries, rlcSineParamsParallelRL } from "../rlc_sine.js";
import { i_at_time_pulse } from "../rl_pulse.js";
import { pulseRlcSolAny } from "./pulse_solvers.js";

function sineParamsAnyLocal(){
  const C = Math.max(0, state.C || 0);
  const top = (state.topology || "seriesRLC");

  if(state.srcMode === "sine" && C > 0){
    if(top === "seriesRLC")      return rlcSineParamsSeries();
    if(top === "C_parallel_RL")  return rlcSineParamsParallelRL();
  }
  return sineParams();
}

function fundamentalPhasor(T, xAt, N=20000){
  // Returns complex PEAK phasor X1 for the 1st harmonic:
  // X1 = (2/T) ∫_0^T x(t) e^{-jωt} dt, ω=2π/T
  const eps = 1e-24;
  const TT = Math.max(eps, T);
  const w = 2*Math.PI / TT;
  const n = Math.max(2000, Math.floor(N));
  const dt = TT / n;

  let re = 0, im = 0;
  for(let k=0;k<n;k++){
    const t = (k + 0.5) * dt;
    const x = xAt(t);
    const ang = w * t;
    re += x * Math.cos(ang);
    im += -x * Math.sin(ang);
  }
  const scale = (2/TT) * dt;
  return { re: re*scale, im: im*scale };
}

function integrateMid(t0, t1, N, fn){
  const a = +t0, b = +t1;
  const n = Math.max(1, Math.floor(N || 1));
  const dt = (b - a) / n;
  let acc = 0;
  for(let k=0;k<n;k++){
    const t = a + (k + 0.5) * dt;
    acc += fn(t);
  }
  return acc * dt;
}

export function powerQualityMetrics(){
  const eps = 1e-12;

  // ---------- SINE ----------
  if(state.srcMode === "sine"){
    const sp = sineParamsAnyLocal();
    const Vrms = sp.Vrms;
    const Irms = sp.Irms;

    const topo = (state.topology || "seriesRLC");
    const topoNorm = (topo === "parallelRLC") ? "C_parallel_RL" : topo;

    let P = 0, Q = 0;

    if(topoNorm === "C_parallel_RL" && isFinite(sp.G) && isFinite(sp.B)){
      // Parallel: Y = G + jB  ->  S = V^2 * conj(Y) = V^2*(G - jB)
      P = (Vrms*Vrms) * sp.G;
      Q = -(Vrms*Vrms) * sp.B;
    }else{
      // Series RL/RLC: real power dissipated only in R
      P = (Irms*Irms) * sp.R;
      Q = Vrms * Irms * Math.sin(sp.phi);
    }

    const S = Vrms * Irms;
    const PF = (S > eps) ? (P / S) : 0;

    const phi = Math.atan2(Q, P);
    const phiDeg = phi * 180/Math.PI;

    return { Vrms, Irms, P, S, PF, Q, phiDeg, phiMeaning: "physical (sine)" };
  }

  // ---------- PULSE (FUNDAMENTAL PQ) ----------
  const sol = pulseRlcSolAny();
  const T = sol ? sol.T : periodPulse();
  const Ton = T * duty();
  const wrap = (t)=> ((t % T) + T) % T;

  const vAt = sol
    ? (t)=> sol.vSrcAt(wrap(t))
    : (t)=> (wrap(t) < Ton ? state.V : 0);

  const iAt = sol
    ? (t)=> sol.iAt(wrap(t))
    : (t)=> i_at_time_pulse(wrap(t));

  // Fundamental (complex PEAK phasors)
  const V1 = fundamentalPhasor(T, vAt, 24000);
  const I1 = fundamentalPhasor(T, iAt, 24000);

  const V1mag = Math.hypot(V1.re, V1.im);
  const I1mag = Math.hypot(I1.re, I1.im);

  const Vrms1 = V1mag / Math.SQRT2;
  const Irms1 = I1mag / Math.SQRT2;

  const P1 = 0.5 * (V1.re * I1.re + V1.im * I1.im);
  const Q1 = 0.5 * (V1.im * I1.re - V1.re * I1.im);
  const S1 = Vrms1 * Irms1;
  const PF1 = (S1 > eps) ? (P1 / S1) : 0;

  const phi = Math.atan2(Q1, P1);
  const phiDeg = phi * 180/Math.PI;

  // Reference/debug “equivalent” values (true RMS + true average P over the period)
  const Vrms_eq = (state.V || 0) * Math.sqrt(Math.max(0, duty()));

  const Irms_eq = Math.sqrt(
    Math.max(
      0,
      integrateMid(0, T, 12000, (t)=>{ const ii=iAt(t); return ii*ii; }) / Math.max(1e-24, T)
    )
  );

  const Egen_per = integrateMid(0, T, 5000, (t)=> vAt(t) * iAt(t));
  const P_eq = Egen_per / Math.max(eps, T);
  const S_eq = Vrms_eq * Irms_eq;
  const PF_eq = (S_eq > eps) ? (P_eq / S_eq) : 0;
  const Q_eq = Math.sqrt(Math.max(0, S_eq*S_eq - P_eq*P_eq));
  const phiEq = Math.acos(Math.max(-1, Math.min(1, PF_eq)));
  const phiDeg_eq = phiEq * 180/Math.PI;

  return {
    Vrms: Vrms1, Irms: Irms1, P: P1, S: S1, PF: PF1, Q: Q1,
    phiDeg, phiMeaning: "fundamental (pulse)",
    // reference / debug
    Vrms_eq, Irms_eq, P_eq, S_eq, PF_eq, Q_eq, phiDeg_eq, phiMeaning_eq: "equivalent (pulse)"
  };
}