// app/metrics/periodicity.js
// Purpose: solver health / validation metric for pulse steady-periodic behavior.
// NOTE: Must NOT import from ../metrics.js to avoid circular deps.

import { state } from "../state.js";
import { periodPulse } from "../core/derived.js";

import { pulseRlcSol, pulseRlcParallelSol } from "./pulse_solvers.js";

// ---------- Pulse periodicity error (steady-state sanity check) ----------
// Goal: quantify how close the solver is to perfect periodic steady-state.
//
// seriesRLC (pulse):
//   Uses the exact 2-state solution (i, vC) and evaluates x(T) - x(0) directly.
//   This should be extremely small (floating-point roundoff).
//
// C_parallel_RL (pulse):
//   The parallel solver wraps tm>=T to modulo, so iAt(T) would always equal iAt(0).
//   We therefore compare i(T-ε) to i(0) (end-of-period vs start). This is a robust
//   "practical periodicity" check for the piecewise RL+edge model.

export function periodicityErrorPulse(){
  const epsT = 1e-12;

  // Only meaningful in pulse mode and only when C>0 and topology is one of the two.
  const C = Math.max(0, state.C || 0);
  if(state.srcMode !== "pulse" || !(C > 0)) return null;

  const top = state.topology || "seriesRLC";
  if(top !== "seriesRLC" && top !== "C_parallel_RL") return null;

  // === SERIES RLC (pulse): exact 2-state check ===
  if(top === "seriesRLC"){
    const sol = pulseRlcSol(); // cached
    if(!sol) return null;

    const T = sol.T || periodPulse();
    const t0 = 0;
    const t1 = T; // IMPORTANT: series solver does NOT modulo inside stateAt(), so this is valid.

    const x0 = sol.stateAt(t0); // [i, vC]
    const x1 = sol.stateAt(t1); // should equal x0 by construction

    const di  = (x1[0] - x0[0]);
    const dvC = (x1[1] - x0[1]);

    const err = Math.hypot(di, dvC);

    return {
      ok: true,
      topology: "seriesRLC",
      T,
      components: { di, dvC },
      errAbs: err,
    };
  }

  // === PARALLEL (R+L)||C (pulse): end-of-period vs start check ===
  // NOTE: parallel solver intentionally wraps tm>=T; so evaluate at T-ε.
  {
    const sol = pulseRlcParallelSol(); // cached
    if(!sol) return null;

    const T = sol.T || periodPulse();
    const tE = Math.max(0, T - Math.max(epsT, 1e-9*T));

    const iRL0 = sol.iRLAt(0);
    const iRLE = sol.iRLAt(tE);

    const i0  = sol.iAt(0);
    const iE  = sol.iAt(tE);

    const vC0 = sol.vCAt(0);
    const vCE = sol.vCAt(tE);

    const vL0 = sol.vLAt(0);
    const vLE = sol.vLAt(tE);

    const diRL = (iRLE - iRL0);
    const di   = (iE   - i0);
    const dvC  = (vCE  - vC0);
    const dvL  = (vLE  - vL0);

    // "practical" error: RL branch current dominates periodic map
    const err = Math.abs(diRL);

    return {
      ok: true,
      topology: "C_parallel_RL",
      T,
      components: { diRL, di, dvC, dvL },
      errAbs: err,
      tProbe: tE,
    };
  }
}