// metrics/measurement.js
// Measurement model (instrument interpretation) — derived from analytic truth.
// NOTE: This is architecture-only extraction from prior metrics.js.
// Behavior and numerical outputs must remain identical.

import { state } from "../state.js";
import { Rtotal, tau, periodPulse, duty } from "../core/derived.js";

// Physics kernels
import { sineParams } from "../rl_sine.js";
import { rlcSineParamsSeries, rlcSineParamsParallelRL } from "../rlc_sine.js";

import {
  steadyI0,
  i_on,
  computeTransientQuantitiesPulse
} from "../rl_pulse.js";

// Metric infrastructure (numeric integration helpers) — local to metrics/
import { integratePulseRlc, _maxAbsCosInterval } from "./integrators.js";

// Solver access (analytic periodic steady-state accessors)
import { pulseRlcSol, pulseRlcParallelSol, pulseRlcSolAny } from "./pulse_solvers.js";

export function sineParamsAny(){
  const C = Math.max(0, state.C || 0);
  const top = (state.topology || "seriesRLC");

  if(state.srcMode === "sine" && C > 0){
    if(top === "seriesRLC")      return rlcSineParamsSeries();
    if(top === "C_parallel_RL")  return rlcSineParamsParallelRL();
  }

  return sineParams();
}

export function windowMagneticMetrics(){
  const eps = 1e-12;
  const L = Math.max(0, state.L);
  const tt = tau();
  const tW = Math.max(0, tt);

  if(tW <= 0 || L <= 0){
    return { Qwin: 0, dLamWin: 0, WLpkWin: 0, dIwin: 0, i0: 0, iT: 0, ELwin: 0 };
  }

  if(state.srcMode === 'sine'){
    const sp = sineParamsAny();
    const w = sp.w, Ipk = sp.Ipk, phi = sp.phi;

    const i0 = Ipk * Math.sin(-phi);
    const iT = Ipk * Math.sin(w*tW - phi);

    const Qwin = (Math.abs(w) > eps)
      ? (Ipk / w) * (Math.cos(phi) - Math.cos(w*tW - phi))
      : (i0 * tW);

    const dIwin = iT - i0;
    const dLamWin = L * dIwin;
    const ELwin = 0.5 * L * (iT*iT - i0*i0);

    let imax = Math.abs(i0);
    const N = 256;
    for(let k=1;k<=N;k++){
      const t = (k/N)*tW;
      const i = Ipk * Math.sin(w*t - phi);
      imax = Math.max(imax, Math.abs(i));
    }
    const WLpkWin = 0.5 * L * (imax*imax);

    return { Qwin, dLamWin, WLpkWin, dIwin, ELwin, i0, iT };
  }

  const solS = pulseRlcSol();
  const solP = pulseRlcParallelSol();
  const sol = solS || solP;

  if(sol){
    const T = sol.T;
    const wrap = (t)=> ((t % T) + T) % T;

    // in parallel: use RL-branch current (inductor current)
    // in series:   use total branch current (same as inductor current)
    const iL = (tm)=> solP ? solP.iRLAt(tm) : sol.iAt(tm);

    const i0 = iL(0);
    const iT = iL(wrap(tW));

    const Qwin = integratePulseRlc(0, tW, 3000, (t)=> iL(wrap(t)));
    const dIwin = iT - i0;
    const dLamWin = L * dIwin;

    // peak |i| in window (sampling)
    const Np = 1200;
    const dtp = tW / Math.max(1, Np);
    let imax = Math.abs(i0);
    for(let k=0;k<Np;k++){
      const t = (k+0.5)*dtp;
      const ii = iL(wrap(t));
      imax = Math.max(imax, Math.abs(ii));
    }
    const WLpkWin = 0.5 * L * (imax*imax);
    const ELwin = 0.5 * L * (iT*iT - i0*i0);

    return { Qwin, dLamWin, WLpkWin, dIwin, ELwin, i0, iT };
  }

  const T = periodPulse();
  const Ton = T * duty();
  const Iinf = state.V / Rtotal();
  const I0 = steadyI0();

  const integrate_i_on = (t)=>{
    // ∫ (Iinf + (I0-Iinf)e^{-t/tt}) dt
    const e = Math.exp(-t/Math.max(eps,tt));
    return Iinf*t + (I0 - Iinf)*(-tt)*(e - 1);
  };

  const iT = (tW <= Ton)
    ? i_on(tW, I0, Iinf, tt)
    : i_on(Ton, I0, Iinf, tt);

  let Qwin = 0;
  if(tW <= Ton + eps){
    Qwin = integrate_i_on(tW);
  }else{
    Qwin = integrate_i_on(Ton);
    const Iend = i_on(Ton, I0, Iinf, tt);
    const u = tW - Ton;
    const e = Math.exp(-u/Math.max(eps,tt));
    Qwin += Iend * tt * (1 - e);
  }

  const dIwin = iT - I0;
  const dLamWin = L * dIwin;

  const iPk = (tW <= Ton) ? i_on(tW, I0, Iinf, tt) : i_on(Ton, I0, Iinf, tt);
  const WLpkWin = 0.5 * L * (iPk*iPk);

  const ELwin = 0.5 * L * (iT*iT - I0*I0);

  return { Qwin, dLamWin, WLpkWin, dIwin, ELwin, i0: I0, iT };
}

export function windowCapacitorMetrics(){
  // Capacitor window metrics over 0 → τ
  // Supported:
  // - seriesRLC:   pulse + sine (existing behavior)
  // - C_parallel_RL: sine only (vC = vSrc; capacitor branch current iC = C·dv/dt)
  const eps = 1e-18;

  const C = Math.max(0, state.C ?? 0);
  if(!(C > 0)) return null;

  const top = state.topology || "seriesRLC";
  if(top !== "seriesRLC" && top !== "C_parallel_RL") return null;

  // pulse: seriesRLC + C_parallel_RL
  if(state.srcMode === "pulse"){
    const solS = (top === "seriesRLC") ? pulseRlcSol() : null;
    const solP = (top === "C_parallel_RL") ? pulseRlcParallelSol() : null;
    const sol = solS || solP;
    if(!sol) return null;

    const T = sol.T;
    const tW = Math.max(0, tau());
    if(!(tW > 0)){
      return {
        VCrmsWin: 0,
        pkAbsVC: 0,
        WCpkWin: 0,
        dQcapWin: 0,
        dWcapWin: 0,
        int_vCi: 0,
        diffW: 0,
        relW: 0,
      };
    }

    const wrap = (t)=> ((t % T) + T) % T;
    const vCAt = (t)=> sol.vCAt(wrap(t));
    const iCAt = (t)=>{
      const tm = wrap(t);
      // series: iC = i (same branch current)
      if(solS) return solS.iAt(tm);
      // parallel: iC = C * dv/dt (solver provides dvSrcAt)
      return C * solP.dvSrcAt(tm);
    };

    const vC0 = vCAt(0);
    const vCT = vCAt(tW);

    const N = 2000;
    const dt = tW / Math.max(1, N);

    let accVC2 = 0;
    let pkAbsVC = 0;
    let acc_vCi = 0;

    for(let k=0;k<N;k++){
      const t = (k + 0.5)*dt;
      const vc = vCAt(t);
      const ic = iCAt(t);

      accVC2 += vc*vc;
      pkAbsVC = Math.max(pkAbsVC, Math.abs(vc));
      acc_vCi += (vc * ic);
    }

    const VCrmsWin = Math.sqrt(accVC2 / N);
    const dQcapWin = C * (vCT - vC0);               // Coulomb
    const dWcapWin = 0.5 * C * (vCT*vCT - vC0*vC0); // Joule
    const int_vCi  = acc_vCi * dt;                  // ∫ vC·iC dt (series => iC=i)

    const diffW = dWcapWin - int_vCi;
    const relW  = Math.abs(diffW) / Math.max(eps, Math.abs(dWcapWin), Math.abs(int_vCi));
    const WCpkWin = 0.5 * C * (pkAbsVC*pkAbsVC);

    return { VCrmsWin, pkAbsVC, WCpkWin, dQcapWin, dWcapWin, int_vCi, diffW, relW };
  }

  // sine (analytic reconstruction)
  if(state.srcMode !== "sine") return null;

  const sp = sineParamsAny();
  const tW = Math.max(0, tau());

  const w   = sp.w;
  const Vpk = state.V;
  const Ipk = sp.Ipk;
  const phi = sp.phi;

  // base waves
  const vSrcAt = (t)=> Vpk * Math.sin(w*t);
  const dvSrcAt = (t)=> w * Vpk * Math.cos(w*t);

  // series current (through everything)
  const iSeriesAt = (t)=> Ipk * Math.sin(w*t - phi);
  const diSeriesAt = (t)=> w * Ipk * Math.cos(w*t - phi);

  // series KVL parts
  const R = Rtotal();
  const vRAt = (t)=> iSeriesAt(t) * R;
  const vLAt = (t)=> state.L * diSeriesAt(t);

  // capacitor voltage depends on topology
  const vCAt = (t)=>{
    if(top === "C_parallel_RL") return vSrcAt(t);        // node voltage in parallel
    return vSrcAt(t) - vRAt(t) - vLAt(t);               // series KVL closure
  };

  // capacitor branch current:
  // - seriesRLC: iC = iSeries
  // - C_parallel_RL: iC = C * dv/dt of node voltage
  const iCAt = (t)=>{
    if(top === "C_parallel_RL") return C * dvSrcAt(t);
    return iSeriesAt(t);
  };

  const vC0 = vCAt(0);
  const vCT = vCAt(tW);

  const N = 1200;
  const dt = tW / Math.max(1, N);

  let accVC2 = 0;
  let pkAbsVC = 0;
  let acc_vCi = 0;

  for(let k=0;k<N;k++){
    const t  = (k + 0.5)*dt;
    const vc = vCAt(t);
    const ic = iCAt(t);

    accVC2 += vc*vc;
    pkAbsVC = Math.max(pkAbsVC, Math.abs(vc));
    acc_vCi += (vc * ic);
  }

  const VCrmsWin = Math.sqrt(accVC2 / N);
  const dQcapWin = C * (vCT - vC0);
  const dWcapWin = 0.5 * C * (vCT*vCT - vC0*vC0);
  const int_vCi  = acc_vCi * dt; // ∫ vC·iC dt (now correct for parallel)

  const diffW = dWcapWin - int_vCi;
  const relW  = Math.abs(diffW) / Math.max(eps, Math.abs(dWcapWin), Math.abs(int_vCi));
  const WCpkWin = 0.5 * C * (pkAbsVC*pkAbsVC);

  return { VCrmsWin, pkAbsVC, WCpkWin, dQcapWin, dWcapWin, int_vCi, diffW, relW };
}

export function impulse_abs_vL_0_to(tWin){
  const tt = tau();
  const tW = Math.max(0, tWin);

  if(state.srcMode === 'sine'){
    const { w, Ipk, phi } = sineParamsAny();
    const A = Math.abs(w*state.L*Ipk);
    const N = 800;
    let acc = 0;
    const dt = tW/Math.max(1, N);
    for(let k=0;k<N;k++){
      const t = (k+0.5)*dt;
      acc += Math.abs(Math.cos(w*t - phi))*dt;
    }
    return A*acc;
  }

  const sol = pulseRlcSolAny();
  if(sol){
    const T = sol.T;
    const wrap = (t)=> ((t % T) + T) % T;
    const N = 5000;
    return integratePulseRlc(0, tW, N, (t)=> Math.abs(sol.vLAt(wrap(t))));
  }

  const T = periodPulse();
  const Ton = T*duty();
  const Toff = T - Ton;
  const Iinf = state.V / Rtotal();
  const I0 = steadyI0();
  const Iend = i_on(Ton, I0, Iinf, tt);

  let remaining = tW;
  let acc = 0;

  const on_coeff = Math.abs(Rtotal()*(Iinf - I0));
  const int_on = (t)=> on_coeff*tt*(1 - Math.exp(-t/tt));

  const off_coeff = Math.abs(Rtotal()*Iend);
  const int_off = (t)=> off_coeff*tt*(1 - Math.exp(-t/tt));

  if(remaining <= Ton){ acc += int_on(remaining); return acc; }
  acc += int_on(Ton); remaining -= Ton;

  if(remaining <= Toff){ acc += int_off(remaining); return acc; }
  acc += int_off(Toff); remaining -= Toff;

  const cyc_int = int_on(Ton) + int_off(Toff);
  const n = Math.floor(remaining / T);
  if(n > 0){ acc += n*cyc_int; remaining -= n*T; }

  if(remaining <= Ton){ acc += int_on(remaining); return acc; }
  acc += int_on(Ton); remaining -= Ton;

  acc += int_off(Math.min(Toff, remaining));
  return acc;
}

export function peak_abs_vL_0_to(tWin){
  const tW = Math.max(0, tWin);

  if(state.srcMode === 'sine'){
    const sp = sineParamsAny();
    const A = Math.abs(sp.w * state.L * sp.Ipk);
    const u0 = -sp.phi;
    const u1 = sp.w * tW - sp.phi;
    const m = _maxAbsCosInterval(u0, u1);
    return A * m;
  }

  const sol = pulseRlcSolAny();
  if(sol){
    const T = sol.T;
    const wrap = (t)=> ((t % T) + T) % T;
    const N = 4000;
    let pk = 0;
    const dt = tW / Math.max(1, N);
    for(let k=0;k<N;k++){
      const t = (k+0.5)*dt;
      pk = Math.max(pk, Math.abs(sol.vLAt(wrap(t))));
    }
    return pk;
  }

  const q = computeTransientQuantitiesPulse();
  return Math.abs(q.vL0);
}