// app/metrics/integrators.js
// Purpose: small, defensible numeric helpers used by multiple metric domains.
// NOTE: Must NOT import from ../metrics.js to avoid circular deps.

import { state } from "../state.js";
import { duty } from "../core/derived.js";
import { pulseRlcSolAny } from "./pulse_solvers.js";

export function integratePulseRlc(t0, t1, N, fn){
  // Midpoint integrator with optional automatic edge refinement for pulse mode.
  // - Keeps the old signature (no call sites changed).
  // - In pulse mode, if a pulse solver exists, it concentrates samples around
  //   the rising/falling edges where dv/dt and di/dt create sharp features.

  const span = Math.max(0, t1 - t0);
  if(span <= 0) return 0;

  // basic midpoint (old behavior)
  function mid(a, b, n){
    const s = Math.max(0, b - a);
    if(s <= 0) return 0;
    const nn = Math.max(1, Math.floor(n));
    const dt = s / nn;
    let acc = 0;
    for(let k=0;k<nn;k++){
      const t = a + (k + 0.5)*dt;
      acc += fn(t);
    }
    return acc * dt;
  }

  // Only refine in pulse mode, when we have a pulse solution with a period.
  if(state.srcMode !== "pulse") return mid(t0, t1, N);

  const sol = pulseRlcSolAny();
  if(!sol || !(sol.T > 0)) return mid(t0, t1, N);

  // We only do "edge segmentation" when the integration window sits inside one period.
  // That matches how your code calls integratePulseRlc: [0..T] or [0..tW] with tW<=T.
  const T = sol.T;
  if(t0 < 0 || t1 > T) return mid(t0, t1, N);

  const Ton = sol.Ton ?? (T * duty());
  if(!(Ton > 0)) return mid(t0, t1, N);

  // Parallel solver provides Tr/Tf; series solver doesn't, so we choose a small default.
  const Tr = Math.max(0, sol.Tr ?? Math.min(0.002*T, 0.05*Ton));
  const Tf = Math.max(0, sol.Tf ?? Tr);

  const riseEnd   = Math.min(Ton, Tr);
  const fallStart = Math.max(0, Ton - Tf);
  const fallEnd   = Ton;

  // Breakpoints inside [t0,t1]
  const pts = [t0];
  const addPt = (x)=>{
    if(x > t0 + 1e-15 && x < t1 - 1e-15) pts.push(x);
  };
  addPt(riseEnd);
  addPt(fallStart);
  addPt(fallEnd);
  pts.push(t1);
  pts.sort((a,b)=>a-b);

  // Build segments and assign weights (boost edges)
  const EDGE_BOOST = 10; // raise this if you want even tighter edge accuracy
  const segs = [];
  for(let i=0;i<pts.length-1;i++){
    const a = pts[i], b = pts[i+1];
    const d = b - a;
    if(d <= 0) continue;

    // segment considered "edge" if it overlaps either edge interval
    const midt = 0.5*(a+b);
    const isEdge = (midt <= riseEnd + 1e-15) || (midt >= fallStart - 1e-15 && midt <= fallEnd + 1e-15);

    const w = d * (isEdge ? EDGE_BOOST : 1);
    segs.push({ a, b, d, w, isEdge });
  }

  if(!segs.length) return 0;

  // Distribute N samples by weight, keep a minimum per segment
  const Ntot = Math.max(1000, Math.floor(N));       // keep decent baseline
  const MIN_PER_SEG = 50;                           // prevents "thin" segments from being ignored
  const totalW = segs.reduce((s,x)=>s+x.w, 0) || 1;

  let Nleft = Ntot;
  for(const s of segs){
    s.n = Math.max(MIN_PER_SEG, Math.floor(Ntot * (s.w / totalW)));
    Nleft -= s.n;
  }

  // fix rounding drift
  if(Nleft !== 0){
    // push remaining samples into edge segments first
    const order = segs.slice().sort((p,q)=> (q.isEdge - p.isEdge) || (q.d - p.d));
    let idx = 0;
    while(Nleft > 0){
      order[idx % order.length].n += 1;
      Nleft -= 1;
      idx++;
    }
    while(Nleft < 0){
      const s = order[idx % order.length];
      if(s.n > MIN_PER_SEG){
        s.n -= 1;
        Nleft += 1;
      }
      idx++;
      if(idx > 100000) break; // safety
    }
  }

  // Integrate segment-wise
  let acc = 0;
  for(const s of segs){
    acc += mid(s.a, s.b, s.n);
  }
  return acc;
}

export function _maxAbsCosInterval(a, b){
  // exact max of |cos(u)| for u in [a,b]
  if(!isFinite(a) || !isFinite(b)) return 0;
  let lo = a, hi = b;
  if(hi < lo){ const t = lo; lo = hi; hi = t; }
  const pi = Math.PI;
  const k0 = Math.ceil(lo / pi);
  const k1 = Math.floor(hi / pi);
  if(k1 >= k0) return 1;
  return Math.max(Math.abs(Math.cos(lo)), Math.abs(Math.cos(hi)));
}