// app/metrics/harmonics.js
// Purpose: pulse-mode harmonic metrics (THD etc) with cache/throttle

import { state } from "../state.js";
import { periodPulse, duty } from "../core/derived.js";
import { pulseRlcSol, pulseTonFromSol } from "./pulse_solvers.js";

function _harmonicPhasor(T, xAt, nH, N=3000){
  // Complex peak phasor for harmonic nH:
  // Xn = (2/T) ∫_0^T x(t) e^{-j nH ω t} dt,  ω=2π/T
  const eps = 1e-24;
  const TT = Math.max(eps, T);
  const w  = 2*Math.PI / TT;

  const n  = Math.max(1200, Math.floor(N));
  const dt = TT / n;

  let re = 0, im = 0;

  // midpoint integration
  for(let k=0;k<n;k++){
    const t = (k + 0.5) * dt;
    const x = xAt(t);
    const ang = (nH * w) * t;
    re += x * Math.cos(ang);
    im += -x * Math.sin(ang);
  }

  const scale = (2/TT) * dt;
  return { re: re*scale, im: im*scale };
}

// --- lightweight cache so we don't recompute spectrum every frame ---
let _hm_cache = null;
let _hm_last_ms = 0;

export function harmonicMetricsPulse(opts = {}){
  // Pulse-mode spectrum (fundamental + THD) for vSrc and i(t)
  // Throttled + lower-N by default for realtime UI.

  if(state.srcMode !== "pulse") return null;

  const now_ms = performance.now();
  const throttle_ms = Math.max(80, Math.min(1000, opts.throttle_ms ?? 250));

  // Return cached result if still "fresh"
  if(_hm_cache && (now_ms - _hm_last_ms) < throttle_ms){
    return _hm_cache;
  }

  const eps = 1e-18;

  const Nh = Math.max(3, Math.min(15, Math.floor(opts.Nh ?? 7)));    // harmonics 1..Nh
  const N  = Math.max(1200, Math.min(12000, Math.floor(opts.N ?? 3000)));

  const sol = pulseRlcSol();
  const T = sol ? sol.T : periodPulse();
  const Ton = sol ? pulseTonFromSol(sol) : (T * duty());
  const wrap = (t)=> ((t % T) + T) % T;

  const vAt = (t)=>{
    const tm = wrap(t);
    if(sol) return sol.vSrcAt(tm);
    // RL fallback: ideal pulse
    return (tm < Ton) ? (state.V || 0) : 0;
  };

  const iAt = (t)=>{
    const tm = wrap(t);
    if(sol) return sol.iAt(tm);
    // RL fallback: use existing time-domain current kernel in rl_pulse.js? (not available here)
    // Keep behavior identical to prior metrics.js: it used pulseRlcSol() only.
    // So if no sol, return 0 spectrum (still stable).
    return 0;
  };

  const V = [];
  const I = [];

  for(let h=1; h<=Nh; h++){
    const Vh = _harmonicPhasor(T, vAt, h, N);
    const Ih = _harmonicPhasor(T, iAt, h, N);

    const Vpk = Math.hypot(Vh.re, Vh.im);
    const Ipk = Math.hypot(Ih.re, Ih.im);

    V.push({ h, Vpk, Vrms: Vpk/Math.SQRT2 });
    I.push({ h, Ipk, Irms: Ipk/Math.SQRT2 });
  }

  const V1 = V[0]?.Vrms ?? 0;
  const I1 = I[0]?.Irms ?? 0;

  let Vh2 = 0, Ih2 = 0;
  for(let k=1;k<V.length;k++) Vh2 += (V[k].Vrms*V[k].Vrms);
  for(let k=1;k<I.length;k++) Ih2 += (I[k].Irms*I[k].Irms);

  const THDv = (V1 > eps) ? Math.sqrt(Vh2) / V1 : 0;
  const THDi = (I1 > eps) ? Math.sqrt(Ih2) / I1 : 0;

  const top = (arr, keyRms, maxItems=3)=>{
    const base = arr[0]?.[keyRms] ?? 0;
    if(!(base > eps)) return [];
    return arr
      .slice(1)
      .map(o=>({ h:o.h, rel:o[keyRms]/base }))
      .sort((a,b)=> b.rel - a.rel)
      .slice(0, maxItems);
  };

  _hm_cache = {
    T,
    harmonicsV: V,
    harmonicsI: I,
    THDv, THDi,
    topV: top(V, "Vrms", 3),
    topI: top(I, "Irms", 3),
  };
  _hm_last_ms = now_ms;

  return _hm_cache;
}