// app/metrics/energy.js
// Purpose: Energy flow + power split metrics (per-period energy bookkeeping).
// NOTE: No imports from ../metrics.js to avoid circular deps.

import { state } from "../state.js";
import { Rtotal, tau, periodPulse, periodSine, duty } from "../core/derived.js";

// Physics kernels (sine steady-state)
import { sineParams } from "../rl_sine.js";
import { rlcSineParamsSeries, rlcSineParamsParallelRL } from "../rlc_sine.js";

// RL pulse helpers (fallback)
import { steadyI0, meanI2_pulse, i_at_time_pulse } from "../rl_pulse.js";

// Cached pulse solvers (series + parallel)
import { pulseRlcSol, pulseRlcParallelSol, pulseRlcSolAny } from "./pulse_solvers.js";

function sineParamsAnyLocal(){
  const C = Math.max(0, state.C || 0);
  const top = (state.topology || "seriesRLC");

  if(state.srcMode === "sine" && C > 0){
    if(top === "seriesRLC")      return rlcSineParamsSeries();
    if(top === "C_parallel_RL")  return rlcSineParamsParallelRL();
  }

  return sineParams();
}

// Local copy (keeps behavior; avoids importing integratePulseRlc from metrics.js)
function integratePulseRlc(t0, t1, N, fn){
  const eps = 1e-24;
  const a = +t0, b = +t1;
  const n = Math.max(1, Math.floor(N || 1));
  const dt = (b - a) / n;

  let acc = 0;
  for(let k=0;k<n;k++){
    const t = a + (k + 0.5) * dt;
    acc += fn(t);
  }

  const sol = pulseRlcSolAny();
  const T = sol ? sol.T : periodPulse();
  const scale = (b - a) / Math.max(eps, T);

  return acc * dt * scale;
}

export function powerSplit(){
  const Rl = Math.max(0, state.Rload);
  const Rc = Math.max(0, state.Rcoil);

  // ---------- SINE ----------
  if(state.srcMode === "sine"){
    const sp = sineParamsAnyLocal();
    const I2 = sp.Irms * sp.Irms;
    const Pload = I2 * Rl;
    const Pcoil = I2 * Rc;
    const Psrc  = Pload + Pcoil;
    return { Psrc, Pload, Pcoil };
  }

  // ---------- PULSE ----------
  const solS = pulseRlcSol();           // seriesRLC (if active)
  const solP = pulseRlcParallelSol();   // C_parallel_RL (if active)

  // series: resistors see total current
  if(solS){
    const I2mean = integratePulseRlc(0, solS.T, 4000, (t)=>{
      const tm = ((t % solS.T) + solS.T) % solS.T;
      const ii = solS.iAt(tm);
      return ii*ii;
    }) / Math.max(1e-12, solS.T);

    const Pload = I2mean * Rl;
    const Pcoil = I2mean * Rc;
    const Psrc  = Pload + Pcoil;
    return { Psrc, Pload, Pcoil };
  }

  // parallel: resistors see ONLY RL-branch current (capacitor current does not dissipate in R)
  if(solP){
    const I2mean_RL = integratePulseRlc(0, solP.T, 4000, (t)=>{
      const tm = ((t % solP.T) + solP.T) % solP.T;
      const ii = solP.iRLAt(tm);
      return ii*ii;
    }) / Math.max(1e-12, solP.T);

    const Pload = I2mean_RL * Rl;
    const Pcoil = I2mean_RL * Rc;
    const Psrc  = Pload + Pcoil;
    return { Psrc, Pload, Pcoil };
  }

  // fallback: original RL pulse
  const I2mean = meanI2_pulse();
  const Pload = I2mean * Rl;
  const Pcoil = I2mean * Rc;
  const Psrc  = Pload + Pcoil;
  return { Psrc, Pload, Pcoil };
}

export function energyFlowMetrics(){
  const eps = 1e-12;
  const L = Math.max(0, state.L);

  const { Psrc, Pload, Pcoil } = powerSplit();

  if(state.srcMode === "sine"){
    const sp = sineParamsAnyLocal();
    const T = periodSine();

    const Egen_per  = Psrc * T;
    const Eload_per = Pload * T;
    const Ecoil_per = Pcoil * T;

    const tt = tau();
    const i0 = sp.Ipk * Math.sin(-sp.phi);
    const i1 = sp.Ipk * Math.sin(sp.w*tt - sp.phi);
    const dWL_tau = 0.5 * L * (i1*i1 - i0*i0);

    const sumR = (Eload_per + Ecoil_per);
    const denom = Math.max(eps, Math.abs(Egen_per), Math.abs(sumR));
    const balanceErr = (Egen_per - sumR) / denom;

    return { T, Egen_per, Eload_per, Ecoil_per, dWL_tau, balanceErr };
  }

  const solS = pulseRlcSol();
  const solP = pulseRlcParallelSol();
  const sol = solS || solP;
  if(sol){
    const T = sol.T;

    const Egen_per = integratePulseRlc(0, T, 5000, (t)=>{
      const tm=((t%T)+T)%T;
      return sol.vSrcAt(tm) * sol.iAt(tm);
    });

    const Rl = Math.max(0, state.Rload);
    const Rc = Math.max(0, state.Rcoil);

    const isParallel = !!solP;
    const iR = (tm)=> isParallel ? sol.iRLAt(tm) : sol.iAt(tm);

    const Eload_per = integratePulseRlc(0, T, 5000, (t)=>{
      const tm=((t%T)+T)%T;
      const ii=iR(tm);
      return ii*ii*Rl;
    });

    const Ecoil_per = integratePulseRlc(0, T, 5000, (t)=>{
      const tm=((t%T)+T)%T;
      const ii=iR(tm);
      return ii*ii*Rc;
    });

    const tt = tau();
    const i0w = sol.iAt(0);
    const i1w = sol.iAt(((tt%T)+T)%T);
    const dWL_tau = 0.5 * L * (i1w*i1w - i0w*i0w);

    const C = Math.max(0, state.C || 0);
    const vC0 = sol.vCAt(0);
    const vC1 = sol.vCAt(((tt%T)+T)%T);
    const dWC_tau = 0.5 * C * (vC1*vC1 - vC0*vC0);

    const dWL_per = 0.5 * L * (sol.iAt(((T%T)+T)%T)**2 - sol.iAt(0)**2);
    // For true periodic steady-state, i(T)=i(0) and vC(T)=vC(0) ⇒ storage deltas ≈ 0.
    const vC_T = sol.vCAt(((T%T)+T)%T);
    const dWC_per = 0.5 * C * (vC_T*vC_T - vC0*vC0);

    const sumR = (Eload_per + Ecoil_per + dWL_per + dWC_per);
    const denom = Math.max(eps, Math.abs(Egen_per), Math.abs(sumR));
    const balanceErr = (Egen_per - sumR) / denom;

    return { T, Egen_per, Eload_per, Ecoil_per, dWL_tau, dWC_tau, dWL_per, dWC_per, balanceErr };
  }

  const Rt = Rtotal();
  const tt = tau();
  const T = periodPulse();
  const Ton = T*duty();
  const Iinf = state.V / Rt;
  const I0 = steadyI0();

  const e1 = Math.exp(-Ton/Math.max(eps,tt));
  const int_i_on = (Iinf*Ton) + (I0 - Iinf)*tt*(1 - e1);
  const Egen_per = state.V * int_i_on;

  const Eload_per = Pload * T;
  const Ecoil_per = Pcoil * T;

  const i0w = i_at_time_pulse(0);
  const i1w = i_at_time_pulse(tt);
  const dWL_tau = 0.5 * L * (i1w*i1w - i0w*i0w);

  const sumR = (Eload_per + Ecoil_per);
  const denom = Math.max(eps, Math.abs(Egen_per), Math.abs(sumR));
  const balanceErr = (Egen_per - sumR) / denom;

  return { T, Egen_per, Eload_per, Ecoil_per, dWL_tau, balanceErr };
}