// main.js
import { state } from "./state.js";
import { bindUI } from "./ui.js";
import {
  initScope,
  resizeScope,
  renderScope,
  getLastScopeData,
  setCursorState,
  clearCursorState,
  setCursorSample,
  clearCursorSample
} from "./scope.js";

import { initXY, renderXY } from "./lissajous.js";
import { initFieldView, renderFieldView, resizeFieldView } from "./field_view.js";
import { initHelixView, renderHelixView, resizeHelixView, clearHelixTrails } from "./helix_view.js";
import { buildWaves } from "./wave_engine.js";

import { createPerfHUD } from "./perf.js";

import { initQuoteLine } from "./quote_line.js";

import { updateAllPanels } from "./panels.js";

import { initImageExport } from "./export_image.js";

const scopeCanvas = document.querySelector("#scope");
const fieldCanvas = document.querySelector("#fieldView");
const helixCanvas = document.querySelector("#helixView");
const viewSel = document.querySelector("#viewMode");

initImageExport({
  stageSel: ".stageTop",
  scopeCanvas,
  fieldCanvas,
  helixCanvas,
  viewSel
});

// ---- Default startup view ----
// Respect a saved preference; otherwise default to SCOPE.
if (viewSel) {
  const saved = localStorage.getItem("viewMode");
  if (saved === "scope" || saved === "field" || saved === "helix") {
    viewSel.value = saved;
  } else {
    viewSel.value = "scope";          // <-- your desired default
    localStorage.setItem("viewMode", "scope");
  }
}

// PERF HUD (toggle by clicking the blue dot)
const perf = createPerfHUD({
  dotEl: document.querySelector(".dot"),
  panelEl: document.querySelector("#perfPanel"),
  sampleWindowMs: 1000,
  targetFrameMs: 1000/60
});

window.__perf = perf; // allow scope.js to tag PHYSICS vs RENDER

// ---------- Formula partial loader ----------
async function loadFormulasPartial(){
  const mount = document.querySelector("#formulaMount");
  if(!mount) return;

  const url = new URL("../partials/formulas.html", import.meta.url).href;

  try{
    const res = await fetch(url, { cache: "no-cache" });
    if(!res.ok) throw new Error(`HTTP ${res.status}`);
    mount.innerHTML = await res.text();
  }catch(err){
    mount.innerHTML = `
      <div class="formulaHead">
        <h3>Formula Reference</h3>
        <p class="formulaSub">Could not load partial: <code>partials/formulas.html</code></p>
      </div>
    `;
    console.warn("Formula partial load failed:", err);
  }
}

initScope(scopeCanvas);
initXY();

initFieldView({
  fieldCanvas,
  scopeCanvas,
  cursorEl: document.querySelector("#cursorReadout"),
  hudEl: document.querySelector(".scopeHud"),
  viewSel
});

initHelixView({ helixCanvas });

bindUI({
  scopeCanvas,
  resizeScope,
  getLastScopeData,
  setCursor: setCursorState,
  clearCursor: clearCursorState,
  setCursorSample,
  clearCursorSample
});

loadFormulasPartial();

initQuoteLine({
  el: document.querySelector("#quoteLine"),
  url: new URL("../partials/marquee.txt", import.meta.url).href,
  pauseBetween: 2000
});

// ---------- Helpers ----------
function getMode(){
  if(viewSel){
    const v = viewSel.value;
    if(v === "field" || v === "helix") return v;
    return "scope";
  }
  const v = localStorage.getItem("viewMode");
  if(v === "field" || v === "helix") return v;
  return "scope";
}

function cssRenderable(c){
  if(!c) return false;
  if(c.offsetParent === null) return false;
  const w = c.clientWidth || 0;
  const h = c.clientHeight || 0;
  return (w >= 120 && h >= 120);
}

function ensureCanvasBacking(c){
  if(!c) return;
  const dpr = Math.min(2, window.devicePixelRatio || 1);
  const rect = c.getBoundingClientRect();
  const bw = Math.max(1, Math.floor(rect.width * dpr));
  const bh = Math.max(1, Math.floor(rect.height * dpr));
  if(c.width !== bw) c.width = bw;
  if(c.height !== bh) c.height = bh;
}

function syncRunUI(){
  const pill = document.querySelector("#statusPill");
  const btn  = document.querySelector("#holdBtn");
  const tagEl = document.querySelector("#hudTag");

  const frozen = !!state.hold;
  const mode = getMode();

  // --- status pill: ALWAYS use the CSS contract (.isLive / .isHold) ---
  if(pill){
    pill.textContent = frozen ? "FROZEN" : "LIVE";
    pill.classList.remove("live", "hold");
    pill.classList.toggle("isHold", frozen);
    pill.classList.toggle("isLive", !frozen);
  }

  // --- hold button label (action) ---
  if(btn){
    btn.textContent = frozen ? "RESUME" : "FREEZE";
    btn.classList.toggle("activeHold", frozen);
    btn.classList.toggle("isFrozen", frozen);
  }

  // --- HUD tag (ONLY update the tag span) ---
  if(tagEl){
    tagEl.textContent =
      (mode === "field") ? "FIELD" :
      (mode === "helix") ? "HELIX" : "SCOPE";
  }
}

// ---------- Field-mode timebase ----------
let runT_s = 0;
let lastNow_s = null;

// watchdog
let lastTstart_seen = null;
let lastAdvanceStamp_s = performance.now() * 1e-3;

// HOLD edge detect
let prevHold = !!state.hold;

// Scope re-prime on view switch
let forceScopePrime = 0;
let scopeCooldown = 0;

window.addEventListener("viewModeChanged", (ev) => {
  const m = ev?.detail?.mode;
  if(m === "scope"){
    forceScopePrime = 8;
    scopeCooldown = 0;
  }
  if(m === "field"){
    try{ resizeFieldView(); }catch(_){}
    lastNow_s = null;
    lastTstart_seen = null;
    lastAdvanceStamp_s = performance.now() * 1e-3;
  }
  if(m === "helix"){
    try{ resizeHelixView(); }catch(_){}
    try{ clearHelixTrails(); }catch(_){}
    lastNow_s = null;
    lastTstart_seen = null;
    lastAdvanceStamp_s = performance.now() * 1e-3;
  }
});

function resetFieldTimebase(){
  runT_s = 0;
  lastNow_s = null;
  lastTstart_seen = null;
  lastAdvanceStamp_s = performance.now() * 1e-3;
}

function buildFrameFor(canvas){
  const span = (state.timeDiv_ms * 1e-3) * 10; // 10 divisions visible
  const now_s = performance.now() * 1e-3;

  if(lastNow_s === null) lastNow_s = now_s;

  if(!state.hold){
    const dtRun = Math.min(0.05, Math.max(0, now_s - lastNow_s));
    runT_s += dtRun;
  }
  lastNow_s = now_s;

  const tStart = runT_s - 0.2 * span;

  const w = Math.max(1, canvas?.clientWidth || 1000);
  const N = Math.max(900, Math.min(4500, Math.floor(w)));

  const built = buildWaves({ tStart, span, N });

  return {
    geom: { x0: 0, y0: 0, w: 1, h: 1 },
    time: { tStart, dt: built.dt, span },
    waves: { vSrc: built.vSrc, i: built.i, vL: built.vL, vC: built.vC }
  };
}

// ---------- Loop ----------
let lastFrame = null;

// Prime once
try{ resizeScope(); }catch(_){}
try{ resizeFieldView(); }catch(_){}
try{ resizeHelixView(); }catch(_){}

if(cssRenderable(scopeCanvas)){
  try{
    renderScope();
    lastFrame = getLastScopeData();
  }catch(_){}
}
if(!lastFrame && cssRenderable(fieldCanvas)){
  ensureCanvasBacking(fieldCanvas);
  try{ resizeFieldView(); }catch(_){}
  try{ lastFrame = buildFrameFor(fieldCanvas); } catch(_){}
}
if(!lastFrame && cssRenderable(helixCanvas)){
  ensureCanvasBacking(helixCanvas);
  try{ resizeHelixView(); }catch(_){}
  try{ lastFrame = buildFrameFor(helixCanvas); } catch(_){}
}

function loop(){
  perf.beginFrame();
  syncRunUI();
  const mode = getMode();

  // UI / DOM housekeeping counts as "render"
  // Panels are updated inside scope.js when in Scope mode.
  // In Field/Helix we must update them here, otherwise tabs go stale/empty.
  if(mode !== "scope"){
    perf.tic("render");
    try { updateAllPanels(); } catch(_){}
    perf.toc();
  }

  const holdNow = !!state.hold;
  if(holdNow !== prevHold){
    prevHold = holdNow;
    if(!holdNow){
      scopeCooldown = 0;
      forceScopePrime = 10;
      resetFieldTimebase();
    }
  }

  // show/hide helix canvas here (field_view.js already handles scope/field)
  if(helixCanvas){
    helixCanvas.style.display = (mode === "helix") ? "block" : "none";
  }

  if(mode === "scope"){
    if(scopeCooldown > 0) scopeCooldown--;

    if(forceScopePrime > 0){
      forceScopePrime--;
      try{ resizeScope(); } catch(_) {}
    }

    if(cssRenderable(scopeCanvas) && scopeCooldown === 0){
      // NOTE: renderScope mixes physics + drawing internally; we count it as "render" here.
      perf.tic("render");
      try{
        renderScope();
        const f = getLastScopeData();
        if(f) lastFrame = f;
      }catch(_){
        scopeCooldown = 10;
      }
      perf.toc();
    }
  } else {
    const activeCanvas = (mode === "field") ? fieldCanvas : helixCanvas;
    if(cssRenderable(activeCanvas)){
      perf.tic("render");
      if(mode === "field"){
        try{ resizeFieldView(); }catch(_){}
      }else{
        try{ resizeHelixView(); }catch(_){}
      }
      ensureCanvasBacking(activeCanvas);
      perf.toc();

      // buildFrameFor() -> buildWaves() => PHYSICS bucket
      perf.tic("physics");
      try{
        const f = buildFrameFor(activeCanvas);
        lastFrame = f;

        // watchdog (fix rare “stays frozen” feeling)
        const ts = f?.time?.tStart;
        const now_s = performance.now() * 1e-3;

        if(ts != null){
          if(lastTstart_seen === null || Math.abs(ts - lastTstart_seen) > 1e-12){
            lastTstart_seen = ts;
            lastAdvanceStamp_s = now_s;
          } else {
            if(!state.hold && (now_s - lastAdvanceStamp_s) > 0.4){
              resetFieldTimebase();
            }
          }
        }
      }catch(_){}
      perf.toc();
    }
  }

  perf.tic("render");
  try { updateMeasStrip(); } catch(_){}
  perf.toc();

  if(lastFrame){
    perf.tic("render");
    try{ renderXY(lastFrame); } catch(_) {}
    try{ renderFieldView(lastFrame); } catch(_) {}
    try{ renderHelixView(lastFrame); } catch(_) {}
    perf.toc();
  }

  perf.endFrame();
  requestAnimationFrame(loop);
}
loop();