// lissajous.js
// XY (Lissajous) renderer using the already computed scope frame data.
// Draws with persistence to mimic oscilloscope XY mode.

import { $ } from "./dom.js";

let ctx = null;
let canvas = null;
let dpr = 1;

let fsX_s = null;
let fsY_s = null;

export function initXY(){
  canvas = $("#xyPlot");
  if(!canvas) return false;
  ctx = canvas.getContext("2d", { alpha:true, desynchronized:true });
  resizeXY();
  window.addEventListener("resize", resizeXY);
  return true;
}

export function resizeXY(){
  if(!canvas) return;
  dpr = Math.min(2, window.devicePixelRatio || 1);
  const rect = canvas.getBoundingClientRect();
  const w = Math.max(1, Math.floor(rect.width * dpr));
  const h = Math.max(1, Math.floor(rect.height * dpr));
  canvas.width = w;
  canvas.height = h;
}

export function clearXY(){
  if(!ctx || !canvas) return;
  const W = canvas.width / dpr;
  const H = canvas.height / dpr;
  ctx.setTransform(dpr,0,0,dpr,0,0);
  ctx.clearRect(0,0,W,H);
  // optionally fill black if you prefer
  ctx.fillStyle = "rgba(0,0,0,1)";
  ctx.fillRect(0,0,W,H);
}

function pickWave(frame, key){
  const wv = frame?.waves;
  if(!wv) return null;
  if(key === "vSrc") return wv.vSrc;
  if(key === "i")    return wv.i;
  if(key === "vL")   return wv.vL;
  if(key === "vC") return wv.vC;
  return null;
}

function colorForKey(key){
  if(key === "vSrc") return "rgba(77,227,255,0.75)";   // CH1 cyan
  if(key === "i")    return "rgba(255,210,77,0.75)";   // CH2 yellow
  if(key === "vL")   return "rgba(178,123,255,0.75)";  // CH3 violet
  if(key === "vC")   return "rgba(120,255,160,0.75)";  // CH4 green
  return "rgba(151,166,178,0.75)";
}

export function renderXY(frame){
  if(!ctx || !canvas || !frame) return;

  const W = canvas.width / dpr;
  const H = canvas.height / dpr;
  ctx.setTransform(dpr,0,0,dpr,0,0);

  const selX = $("#xyX");
  const selY = $("#xyY");
  const pers = $("#xyPersist");

  const xKey = selX?.value || "vSrc";
  const yKey = selY?.value || "i";
  const persist = Math.max(0, Math.min(0.98, parseFloat(pers?.value ?? "0.85")));

  const X = pickWave(frame, xKey);
  const Y = pickWave(frame, yKey);
  if(!X || !Y || X.length !== Y.length) return;

  // persistence fade
  ctx.save();
  ctx.globalCompositeOperation = "source-over";
  ctx.fillStyle = `rgba(0,0,0,${1 - persist})`;
  ctx.fillRect(0,0,W,H);
  ctx.restore();

  // grid + border (lightweight)
  const pad = 18;
  const bottomPadExtra = 38;      // reserve space for the preset bar

  const x0 = pad;
  const y0 = pad;
  const w  = W - 2*pad;
  const h  = H - 2*pad - bottomPadExtra;  // <-- keep trace above the buttons

  const midX = x0 + w/2;
  const midY = y0 + h/2;

  ctx.save();
  ctx.strokeStyle = "rgba(30,42,58,.55)";
  ctx.lineWidth = 1;

  // border
  ctx.strokeRect(x0, y0, w, h);

  // crosshair
  ctx.strokeStyle = "rgba(151,166,178,.18)";
  ctx.beginPath();
  ctx.moveTo(midX, y0); ctx.lineTo(midX, y0+h);
  ctx.moveTo(x0, midY); ctx.lineTo(x0+w, midY);
  ctx.stroke();
  ctx.restore();

  // autoscale (EMA to avoid breathing)
  let fsX = 1e-9, fsY = 1e-9;
  for(let k=0;k<X.length;k++){
    fsX = Math.max(fsX, Math.abs(X[k]));
    fsY = Math.max(fsY, Math.abs(Y[k]));
  }
  fsX *= 1.15;
  fsY *= 1.15;

  const alpha = 0.12;
  if(fsX_s === null) fsX_s = fsX;
  if(fsY_s === null) fsY_s = fsY;
  fsX_s = fsX_s + alpha*(fsX - fsX_s);
  fsY_s = fsY_s + alpha*(fsY - fsY_s);

  const sx = (w/2) / Math.max(1e-12, fsX_s);
  const sy = (h/2) / Math.max(1e-12, fsY_s);

  // draw XY trace
  ctx.save();
  ctx.globalCompositeOperation = "lighter";
  ctx.lineWidth = 1.6;
  //ctx.strokeStyle = "rgba(77,227,255,0.75)"; // reuse CH1-like cyan for readability
  ctx.strokeStyle = colorForKey(yKey); // or mix xKey/yKey; this is simplest

  ctx.beginPath();
  for(let k=0;k<X.length;k++){
    const px = midX + X[k]*sx;
    const py = midY - Y[k]*sy;
    if(k===0) ctx.moveTo(px,py); else ctx.lineTo(px,py);
  }
  ctx.stroke();
  ctx.restore();
}
