// helix_view.js  (CORRECT / SCOPE-FAITHFUL + ORBIT)
// Helix meaning: (x,y,z) = (vSrc(t), i(t), windowDepth(t)) with a camera projection.
// Key rule: connect ONLY consecutive time samples. No strands/stride.

import { $ } from "./dom.js";
import { state } from "./state.js";

let canvas = null;
let ctx = null;
let dpr = 1;

let trail = null;
let tctx = null;

let _bw = 0, _bh = 0;

// smoothed autoscale
let fsX_s = null;
let fsY_s = null;

// fps + orbit time
let _orbitT = 0;        // seconds
let _orbitLast = 0;     // ms
let _fpsEma = 60;

let _helixStateHash = "";

export function initHelixView(opts = {}){
  canvas = opts.helixCanvas || $("#helixView");
  if(!canvas) return false;

  canvas.style.position = "absolute";
  canvas.style.inset = "0";
  canvas.style.width = "100%";
  canvas.style.height = "100%";
  canvas.style.zIndex = "2";
  canvas.style.pointerEvents = "none";

  ctx = canvas.getContext("2d", { alpha:true, desynchronized:true });

  trail = document.createElement("canvas");
  tctx = trail.getContext("2d", { alpha:true, desynchronized:true });

  // expose persistence buffer for exporter
  canvas.__trailCanvas = trail;

  resizeHelixView();
  window.addEventListener("resize", resizeHelixView);
  return true;
}

export function resizeHelixView(){
  if(!canvas) return;
  dpr = Math.min(2, window.devicePixelRatio || 1);
  const rect = canvas.getBoundingClientRect();
  const w = Math.max(1, Math.floor(rect.width * dpr));
  const h = Math.max(1, Math.floor(rect.height * dpr));
  if(w === _bw && h === _bh) return;
  _bw = w; _bh = h;

  canvas.width = w;
  canvas.height = h;

  if(trail){
    trail.width = w;
    trail.height = h;
  }
  clearHelixTrails();
}

export function clearHelixTrails(){
  if(!tctx || !trail || !canvas) return;
  const W = canvas.width / dpr;
  const H = canvas.height / dpr;
  tctx.setTransform(dpr,0,0,dpr,0,0);
  tctx.clearRect(0,0,W,H);
}

function knob(id, def, min, max){
  const el = $(id);
  let v = def;
  if(el){
    const p = parseFloat(el.value);
    if(isFinite(p)) v = p;
  }
  if(min != null) v = Math.max(min, v);
  if(max != null) v = Math.min(max, v);
  return v;
}

// Use same slider, but clamp lower for helix so it doesn’t turn into a web
function helixPersist(){
  const p = knob("#fieldPersist", 0.78, 0, 0.98);
  const t = p / 0.98;
  const eased = 1 - Math.pow(1 - t, 3.2);
  return Math.min(0.86, eased * 0.98);
}

function helixGain(){
  // optional UI slider later: <input id="helixGain" type="range" ...>
  // works even if it doesn't exist (defaults to 1.0)
  let v = 1.0;

  const el = $("#helixGain");
  if(el){
    const p = parseFloat(el.value);
    if(isFinite(p)) v = p;
  } else {
    const ls = parseFloat(localStorage.getItem("helixGain") || "");
    if(isFinite(ls)) v = ls;
  }

  return Math.max(0.25, Math.min(5.0, v));
}

function periodFromState(){
  const eps = 1e-12;
  if(state.srcMode === "pulse"){
    const f = Math.max(eps, state.f || 0);
    return 1 / f;
  } else {
    const f = Math.max(eps, state.fs || 0);
    return 1 / f;
  }
}

function pickWave(frame, key){
  const wv = frame?.waves;
  if(!wv) return null;
  if(key === "vSrc") return wv.vSrc;
  if(key === "i")    return wv.i;
  if(key === "vL")   return wv.vL;
  if(key === "vC")   return wv.vC;
  return null;
}

function rot2(x, y, ang){
  const c = Math.cos(ang), s = Math.sin(ang);
  return { x: x*c - y*s, y: x*s + y*c };
}

// Camera projection (yaw + pitch + perspective)
function projectCam(x, y, z, cam){
  const cy = Math.cos(cam.yaw),  sy = Math.sin(cam.yaw);
  const cx = Math.cos(cam.pitch), sx = Math.sin(cam.pitch);

  // yaw around Y (x,z)
  let x1 =  x*cy + z*sy;
  let z1 = -x*sy + z*cy;

  // pitch around X (y,z)
  let y2 =  y*cx - z1*sx;
  let z2 =  y*sx + z1*cx;

  // perspective (near clamp to avoid “inside” explosions)
  const denom = Math.max(280, cam.dist + z2);
  const s = cam.dist / denom;

  return { x: x1*s, y: y2*s, s, z: z2 };
}

// HSV -> RGB (fast, no allocations)
function hsv2rgb(h, s, v){
  h = ((h % 360) + 360) % 360;
  const c = v * s;
  const x = c * (1 - Math.abs(((h / 60) % 2) - 1));
  const m = v - c;

  let rp=0, gp=0, bp=0;
  if(h < 60){ rp=c; gp=x; bp=0; }
  else if(h < 120){ rp=x; gp=c; bp=0; }
  else if(h < 180){ rp=0; gp=c; bp=x; }
  else if(h < 240){ rp=0; gp=x; bp=c; }
  else if(h < 300){ rp=x; gp=0; bp=c; }
  else { rp=c; gp=0; bp=x; }

  return {
    r: Math.floor((rp + m) * 255),
    g: Math.floor((gp + m) * 255),
    b: Math.floor((bp + m) * 255)
  };
}

// Color bands (center brighter, outer more saturated, full hue sweep)
function bandColor(b, bands){
  const u = (bands <= 1) ? 0 : (b / (bands - 1)); // 0..1

  // Hue sweep (avoid wrapping through pure red if you dislike it)
  // 210°(blue) -> 30°(orange) gives a nice "plasma" look
  const hue = 210 - 180 * u;

  // Bright core, more saturation outward
  const s = 0.35 + 0.55 * u;        // 0.35..0.90
  const v = 0.92 - 0.18 * u;        // 0.92..0.74

  const rgb = hsv2rgb(hue, s, v);

  // Alpha: keep center strongest (readability) + softer outer glow
  const a = 0.50 - 0.22 * u;        // 0.50..0.28

  return { r: rgb.r, g: rgb.g, b: rgb.b, a };
}

function updateOrbit(){
  const now = performance.now();
  if(_orbitLast > 0){
    const dtMs = Math.max(1, now - _orbitLast);
    const fps = 1000 / dtMs;
    _fpsEma = _fpsEma + 0.08*(fps - _fpsEma);

    const dt = Math.min(0.05, dtMs * 1e-3);
    _orbitT += dt;
  } else {
    _fpsEma = 60;
  }
  _orbitLast = now;
}

export function renderHelixView(frame){
  if(!ctx || !canvas || !frame || !tctx || !trail) return;

  const mode = $("#viewMode")?.value || localStorage.getItem("viewMode") || "scope";
  if(mode !== "helix") return;

  // Detect major physics changes and reset trails
  const newHash = `${state.srcMode}_${state.R}_${state.L}_${state.C}_${state.f}_${state.fs}`;
  if(newHash !== _helixStateHash){
    clearHelixTrails();
    _helixStateHash = newHash;
  }

  // Respect scope channel toggles (HELIX uses v(t) vs i(t))
  const chV = $("#chV")?.checked ?? true; // CH1
  const chI = $("#chI")?.checked ?? true; // CH2
  if(!chV || !chI){
    const W = canvas.width / dpr;
    const H = canvas.height / dpr;
    ctx.setTransform(dpr,0,0,dpr,0,0);
    ctx.clearRect(0,0,W,H);          // clear visible
    clearHelixTrails();              // clear trail buffer
    return;
  }

  updateOrbit();

  const W = canvas.width / dpr;
  const H = canvas.height / dpr;

  ctx.setTransform(dpr,0,0,dpr,0,0);
  tctx.setTransform(dpr,0,0,dpr,0,0);

  // Pair (scope-faithful: use the same signals)
  const xKey = "vSrc";
  const yKey = "i";

  const X = pickWave(frame, xKey);
  const Y = pickWave(frame, yKey);
  if(!X || !Y || X.length !== Y.length || X.length < 16){
    ctx.clearRect(0,0,W,H);
    clearHelixTrails();
    return;
  }

  // Persistence fade (trail buffer)
  const persist = helixPersist();
  tctx.save();
  tctx.globalCompositeOperation = "destination-out";
  tctx.fillStyle = `rgba(0,0,0,${1 - persist})`;
  tctx.fillRect(0,0,W,H);
  tctx.restore();

  // Clear visible canvas
  ctx.clearRect(0,0,W,H);

  // Layout
  const pad = 18;
  const x0 = pad, y0 = pad;
  const w  = W - 2*pad, h = H - 2*pad;
  const cx = x0 + w/2, cy = y0 + h/2;

  // Subtle grid box
  ctx.save();
  ctx.strokeStyle = "rgba(30,42,58,.70)";
  ctx.lineWidth = 1;
  ctx.strokeRect(x0,y0,w,h);
  ctx.strokeStyle = "rgba(151,166,178,.18)";
  ctx.beginPath();
  ctx.moveTo(cx, y0); ctx.lineTo(cx, y0+h);
  ctx.moveTo(x0, cy); ctx.lineTo(x0+w, cy);
  ctx.stroke();
  ctx.restore();

  // Autoscale (EMA) based on the same data used in scope
  let fsX = 1e-9, fsY = 1e-9;
  for(let k=0;k<X.length;k++){
    const ax = Math.abs(X[k]);
    const ay = Math.abs(Y[k]);
    if(ax > fsX) fsX = ax;
    if(ay > fsY) fsY = ay;
  }
  fsX *= 1.15; fsY *= 1.15;

  const emaA = 0.12;
  if(fsX_s === null) fsX_s = fsX;
  if(fsY_s === null) fsY_s = fsY;
  fsX_s = fsX_s + emaA*(fsX - fsX_s);
  fsY_s = fsY_s + emaA*(fsY - fsY_s);

  const sx = (w*0.48) / Math.max(1e-12, fsX_s);
  const sy = (h*0.48) / Math.max(1e-12, fsY_s);

  // Time / depth
  const tStart = frame?.time?.tStart ?? 0;
  const dt = frame?.time?.dt ?? 0;
  const span = frame?.time?.span ?? (dt * (X.length-1));
  const TT = Math.max(1e-12, periodFromState());

  // ---- MEANING + VIEW ----
  // z is local window depth (0..zMax) + constant offset so we stay in front of camera
  const zMax = 1.35;
  const zOffset = 0.35;

  // mild phase twist (optional)
  const twist = 0.25;

  // Orbit camera (gentle + human-ish)
  const orbitSpeed = 0.05;  // slower
  const orbitAmp   = 0.45;
  const baseYaw    = 0.20;

  const basePitch  = 0.38;
  const pitchBob   = 0.02;

  const cam = {
    pitch: -(basePitch + pitchBob*Math.sin(_orbitT*0.6)),
    yaw:   baseYaw + orbitAmp*Math.sin(_orbitT*orbitSpeed*2*Math.PI),
    zScale: 420,
    dist:   3600
  };

  // Downsample ONLY uniformly (keeps time continuity)
  const fps = Math.max(5, Math.min(120, _fpsEma));
  const perfScale = Math.max(0.60, Math.min(1.0, fps / 60));

  const bands = Math.max(6, Math.min(10, Math.round(9 * perfScale)));
  const maxSegments = Math.round(1400 * perfScale);
  const N = X.length;
  const step = Math.max(1, Math.floor(N / Math.max(600, maxSegments)));
  const gain = helixGain();
  const rMax = Math.max(1e-9, Math.max(fsX_s, fsY_s));

  // Build per-band segment lists (BUT segments are always consecutive in time!)
  const segs = Array.from({ length: bands }, () => []);

  let prevXX = null, prevYY = null;
  let tipXX = null, tipYY = null, tipBand = 0;

  for(let k=0;k<N;k+=step){
    const t = tStart + k*dt;

    const uWin = (span > 0) ? ((t - tStart) / span) : (k/(N-1));
    const u = Math.max(0, Math.min(1, uWin));
    const z = zMax * u + zOffset;

    // -------- FIELD VORTEX (state vector polar form) --------
    // normalize i so v and i contribute similarly
    const kI = (fsY_s > 1e-12 && fsX_s > 1e-12) ? (fsX_s / fsY_s) : 1;

    // v and i in comparable units
    const vN = X[k];
    const iN = kI * Y[k];

    // polar coords of the state vector
    const theta = Math.atan2(iN, vN);
    const di = (k>0 && dt > 0) ? (Y[k] - Y[k-1]) / dt : 0;
    const r = Math.hypot(vN, iN) * (1 + 0.25*Math.tanh(di));

    // map radius to pixels (based on v-scale)
    const rScale = ((Math.min(w, h) * 0.48) / Math.max(1e-12, fsX_s)) * gain;

    // back to cartesian in “field plane”
    const depthFactor = 1 - 0.6*u;   // narrower at far end
    let px = (r * rScale * depthFactor) * Math.cos(theta);
    let py = (r * rScale * depthFactor) * Math.sin(theta);

    // optional gentle twist
    if(twist !== 0){
      const phase = 2*Math.PI * (t / TT);
      const rr = rot2(px, py, phase * twist);
      px = rr.x; py = rr.y;
    }

    const p = projectCam(px, py, z * cam.zScale, cam);

    const xx = cx + p.x;
    const yy = cy + p.y - h*0.14;   // move helix upward (tune 0.08..0.22)

    if(prevXX === null){
      prevXX = xx; prevYY = yy;
      continue;
    }

    // segment band by signal radius (stable in signal space)
    const radiusNorm = Math.min(1, Math.hypot(X[k], Y[k]) / rMax);
    const band = Math.max(0, Math.min(bands-1, Math.floor(radiusNorm * (bands-1))));
    tipXX = xx;
    tipYY = yy;
    tipBand = band;

    // store this consecutive segment into that band
    segs[band].push(prevXX, prevYY, xx, yy);

    prevXX = xx; prevYY = yy;
  }

  // Draw segments per band (batched, but NO chord-mess)
  tctx.save();
  tctx.globalCompositeOperation = "lighter";
  tctx.lineWidth = 1.55;
  tctx.shadowBlur = 8;

  for(let b=0;b<bands;b++){
    const pts = segs[b];
    if(pts.length < 4) continue;

    const c = bandColor(b, bands);
    tctx.strokeStyle = `rgba(${c.r},${c.g},${c.b},${c.a})`;
    tctx.shadowColor = `rgba(${c.r},${c.g},${c.b},0.9)`;

    tctx.beginPath();
    for(let i=0;i<pts.length;i+=4){
      tctx.moveTo(pts[i],   pts[i+1]);
      tctx.lineTo(pts[i+2], pts[i+3]);
    }
    tctx.stroke();
  }

  tctx.restore();

  // Composite trails onto visible canvas
  ctx.save();
  ctx.globalCompositeOperation = "lighter";
  ctx.drawImage(trail, 0, 0, W, H);
  ctx.restore();

  // Moving phase tip marker (newest sample)
  if(tipXX != null && tipYY != null){
    const c = bandColor(tipBand, bands);

    ctx.save();
    ctx.globalCompositeOperation = "lighter";

    const rDot = Math.max(2.6, Math.min(6.0, 3.4 * Math.sqrt(gain)));

    ctx.shadowBlur = 14;
    ctx.shadowColor = `rgba(${c.r},${c.g},${c.b},0.95)`;
    ctx.fillStyle   = `rgba(${c.r},${c.g},${c.b},0.95)`;

    ctx.beginPath();
    ctx.arc(tipXX, tipYY, rDot, 0, Math.PI * 2);
    ctx.fill();

    // tiny bright core
    ctx.shadowBlur = 0;
    ctx.fillStyle = "rgba(255,255,255,0.85)";
    ctx.beginPath();
    ctx.arc(tipXX, tipYY, Math.max(1.2, rDot*0.38), 0, Math.PI * 2);
    ctx.fill();

    ctx.restore();
  }

  // Bottom-left HELIX info (auto-sized + truthful)
  ctx.save();

  const lineH = 16;
  const padX = 12;
  const padY = 10;

  ctx.font = "12px system-ui, sans-serif";

  // values that are guaranteed in HELIX render scope
  const fpsTxt     = (typeof fps === "number" && isFinite(fps)) ? fps.toFixed(0) : "—";
  const persistTxt = (typeof persist === "number" && isFinite(persist)) ? persist.toFixed(2) : "—";

  // optional extras if present (won't crash if not)
  const Ntxt   = (typeof N === "number" && isFinite(N)) ? `${N}` : "—";
  const stepTxt = (typeof step === "number" && isFinite(step)) ? `${step}` : "—";
  const segTxt  = (typeof maxSegments === "number" && isFinite(maxSegments)) ? `${maxSegments}` : "—";
  const bandsTxt = (typeof bands === "number" && isFinite(bands)) ? `${bands}` : "—";

  // if you have autoscale variables in your file, show them; otherwise "—"
  const sxTxt = (typeof sx === "number" && isFinite(sx)) ? sx.toFixed(3) : "—";
  const syTxt = (typeof sy === "number" && isFinite(sy)) ? sy.toFixed(3) : "—";

  const textLines = [
    "HELIX · State Vector Vortex (v,i)",
    "r = √(v² + (k·i)²)  ·  θ = atan2(k·i, v)",
    `N=${Ntxt}  ·  step=${stepTxt}  ·  seg=${segTxt}`,
    `bands=${bandsTxt}  ·  persist=${persistTxt}  ·  fps≈${fpsTxt}`,
    `scale: sx=${sxTxt}  ·  sy=${syTxt}`
  ];

  // measure width
  let maxW = 0;
  for(const s of textLines){
    const mw = ctx.measureText(s).width;
    if(mw > maxW) maxW = mw;
  }

  // box size + clamp to viewport
  const boxW = Math.min(maxW + padX*2, w - 20);
  const boxH = padY*2 + textLines.length * lineH;

  // box position (bottom-left)
  const boxX = x0 + 10;
  const boxY = y0 + h - boxH - 10;

  // background
  ctx.globalAlpha = 0.72;
  ctx.fillStyle = "rgba(12,18,26,0.86)";
  ctx.fillRect(boxX, boxY, boxW, boxH);

  // text
  ctx.globalAlpha = 1;
  ctx.fillStyle = "rgba(230,237,243,0.95)";

  let ty = boxY + padY + lineH - 2;
  for(const s of textLines){
    ctx.fillText(s, boxX + padX, ty);
    ty += lineH;
  }

  ctx.restore();
}