// field_view.js
// FIELD VIEW: Stable multi-trace "Energy Chamber" overlay.
// Key fixes:
//  1) Persistence uses an offscreen trail buffer.
//  2) resizeFieldView() only resizes when size actually changes (prevents trail wipe every frame).
//  3) BUT it resizes BOTH main canvas + trail canvas and clears trail on resize / on enter field.

import { $ } from "./dom.js";
import { state } from "./state.js";

let canvas = null;
let ctx = null;
let dpr = 1;

let _scopeCanvas = null;
let _cursorEl = null;
let _hudEl = null;
let _viewSel = null;

// Offscreen trail buffer (persistence lives here)
let trail = null;
let tctx = null;

// last backing size (device px)
let _bw = 0;
let _bh = 0;

// Per-trace autoscale smoothing
const _scale = {
  vSrc_i: { fsX: null, fsY: null },
  vSrc_vL: { fsX: null, fsY: null },
  vSrc_vC: { fsX: null, fsY: null },
  i_vC: { fsX: null, fsY: null }
};

// energy smoothing
let E_s = 0;
let dE_s = 0;
let lastEraw = 0;
let lastT = null;

export function initFieldView(opts = {}) {
  canvas = opts.fieldCanvas || $("#fieldView");
  if (!canvas) return false;

  _scopeCanvas = opts.scopeCanvas || $("#scope");
  _cursorEl = opts.cursorEl || $("#cursorReadout");
  _hudEl = opts.hudEl || document.querySelector(".scopeHud");
  _viewSel = opts.viewSel || $("#viewMode");

  canvas.style.position = "absolute";
  canvas.style.inset = "0";
  canvas.style.width = "100%";
  canvas.style.height = "100%";
  canvas.style.zIndex = "2";
  canvas.style.pointerEvents = "none";

  ctx = canvas.getContext("2d", { alpha: true, desynchronized: true });

  trail = document.createElement("canvas");
  tctx = trail.getContext("2d", { alpha: true, desynchronized: true });

  // expose persistence buffer for exporter
  canvas.__trailCanvas = trail;

  // initial sizing (may be hidden -> 0px; we guard and resize again on mode switch)
  resizeFieldView();

  window.addEventListener("resize", resizeFieldView);

  if (_viewSel) {
    const saved = localStorage.getItem("viewMode");
    if (saved === "field" || saved === "scope" || saved === "helix") _viewSel.value = saved;

    applyViewMode(_viewSel.value);

    _viewSel.addEventListener("change", () => {
      const v = (_viewSel.value === "field") ? "field"
              : (_viewSel.value === "helix") ? "helix"
              : "scope";
      localStorage.setItem("viewMode", v);
      applyViewMode(v);
      requestAnimationFrame(() => resizeFieldView());
    });
  }

  return true;
}

function applyViewMode(mode) {
  const isField = (mode === "field");
  const isHelix = (mode === "helix");

  // Scope only when "scope"
  if (_scopeCanvas) _scopeCanvas.style.display = (!isField && !isHelix) ? "block" : "none";

  // Field only when "field"
  if (canvas) canvas.style.display = isField ? "block" : "none";

  // Cursor readout only makes sense for Scope
  if (_cursorEl) _cursorEl.style.display = (!isField && !isHelix) ? "" : "none";

  // keep HUD always
  if (_hudEl) _hudEl.style.display = "";

  if (isField) {
    requestAnimationFrame(() => {
      resizeFieldView();
      clearTrails();
    });
  }

  window.dispatchEvent(new CustomEvent("viewModeChanged", {
    detail: { mode: isField ? "field" : (isHelix ? "helix" : "scope") }
  }));
}

function clearTrails() {
  if (!tctx || !trail) return;
  const W = canvas ? (canvas.width / dpr) : (trail.width / dpr);
  const H = canvas ? (canvas.height / dpr) : (trail.height / dpr);
  tctx.save();
  tctx.setTransform(dpr, 0, 0, dpr, 0, 0);
  tctx.globalCompositeOperation = "source-over";
  tctx.clearRect(0, 0, W, H);
  tctx.restore();
}

export function resizeFieldView() {
  if (!canvas) return;

  dpr = Math.min(2, window.devicePixelRatio || 1);
  const rect = canvas.getBoundingClientRect();

  const w = Math.max(1, Math.floor(rect.width * dpr));
  const h = Math.max(1, Math.floor(rect.height * dpr));

  // IMPORTANT: do not reassign width/height if unchanged (it clears the bitmap!)
  if (w === _bw && h === _bh) return;
  _bw = w; _bh = h;

  canvas.width = w;
  canvas.height = h;

  if (trail) {
    trail.width = w;
    trail.height = h;
  }

  // Clear trails after a true resize to avoid stale scaling artifacts
  clearTrails();
}

// -------- knobs --------
function knob(id, def, min, max){
  const el = $(id);
  let v = def;
  if (el) {
    const p = parseFloat(el.value);
    if (isFinite(p)) v = p;
  }
  if (min != null) v = Math.max(min, v);
  if (max != null) v = Math.min(max, v);
  return v;
}

function persistValue(){
  // perceptual mapping so slider feels useful across full travel
  const p = knob("#fieldPersist", 0.90, 0, 0.98);   // UI 0..0.98
  const t = p / 0.98;                               // 0..1
  const eased = 1 - Math.pow(1 - t, 3.2);           // ease-out
  return eased * 0.98;
}

function glowValue(){     return knob("#fieldGlow", 0.85, 0, 1.8); }
function dotValue(){      return knob("#fieldDot", 0.95, 0, 1.8); }
function depthValue(){    return knob("#fieldDepth", 0.95, 0, 1.8); }
function rotValue(){      return knob("#fieldRotate", 0.55, 0, 2.5); }
function energyColorOn(){ return knob("#fieldEnergyColor", 1, 0, 1) > 0.5; }
function fogValue(){      return knob("#fieldFog", 1.00, 0, 2.0); }
function vignetteValue(){ return knob("#fieldVig", 1.00, 0, 2.0); }
function shimmerValue(){  return knob("#fieldShimmer", 0.70, 0, 2.0); }

// -------- colors --------
const COL_CH2 = [255, 210,  77];
const COL_CH3 = [178, 123, 255];
const COL_CH4 = [120, 255, 160];

function rgba(rgb, a){ return `rgba(${rgb[0]},${rgb[1]},${rgb[2]},${a})`; }
function mix(a, b, t){
  t = Math.max(0, Math.min(1, t));
  return [
    Math.round(a[0] + (b[0]-a[0])*t),
    Math.round(a[1] + (b[1]-a[1])*t),
    Math.round(a[2] + (b[2]-a[2])*t),
  ];
}

// -------- energy helpers --------
function energyMixFactor(i, vC){
  const L = Math.max(0, state.L || 0);
  const C = Math.max(0, state.C || 0);
  const WL = L * (i*i);
  const WC = C * (vC*vC);
  const den = WL + WC;
  if (den <= 0) return 0.5;
  return WC / den;
}
function energyTotal(i, vC){
  const L = Math.max(0, state.L || 0);
  const C = Math.max(0, state.C || 0);
  return (L * (i*i)) + (C * (vC*vC));
}
function energyBalance(i, vC){
  const L = Math.max(0, state.L || 0);
  const C = Math.max(0, state.C || 0);
  const WL = L * (i*i);
  const WC = C * (vC*vC);
  const s = WL + WC;
  if (s <= 0) return 0.0;
  const d = Math.abs(WL - WC) / s;
  return 1 - d;
}

// -------- background --------
function drawEnergyFog(W, H, midX, midY, E_norm, bal_norm, shimmer){
  const fog = fogValue();
  const vig = vignetteValue();

  const fogA  = (0.10 + 0.42 * E_norm) * fog;
  const coreA = (0.06 + 0.22 * bal_norm) * fog;
  const rippleA = 0.10 * shimmer * fog;

  const r0 = Math.min(W, H) * (0.06 + 0.10 * bal_norm);
  const r1 = Math.min(W, H) * (0.55 + 0.20 * E_norm);

  const g = ctx.createRadialGradient(midX, midY, r0, midX, midY, r1);
  g.addColorStop(0.0, `rgba(255,255,255,${coreA})`);
  g.addColorStop(0.25, `rgba(140,220,190,${fogA * 0.55})`);
  g.addColorStop(0.55, `rgba(120,140,255,${fogA * 0.35})`);
  g.addColorStop(1.0, `rgba(0,0,0,0)`);

  ctx.save();
  ctx.globalCompositeOperation = "lighter";
  ctx.fillStyle = g;
  ctx.fillRect(0, 0, W, H);
  ctx.restore();

  if(rippleA > 0.001){
    const t = performance.now() * 1e-3;
    const rr = Math.min(W, H) * (0.18 + 0.08 * Math.sin(t * 1.7));
    ctx.save();
    ctx.globalCompositeOperation = "lighter";
    ctx.strokeStyle = `rgba(200,255,230,${rippleA})`;
    ctx.lineWidth = 2.0;
    ctx.beginPath();
    ctx.arc(midX, midY, rr, 0, Math.PI * 2);
    ctx.stroke();
    ctx.restore();
  }

  const vg = ctx.createRadialGradient(
    midX, midY, Math.min(W, H)*0.20,
    midX, midY, Math.min(W, H)*0.85
  );
  vg.addColorStop(0, "rgba(0,0,0,0)");
  vg.addColorStop(0.70, `rgba(0,0,0,${0.10*vig})`);
  vg.addColorStop(1.00, `rgba(0,0,0,${0.42*vig})`);

  ctx.save();
  ctx.globalCompositeOperation = "source-over";
  ctx.fillStyle = vg;
  ctx.fillRect(0, 0, W, H);
  ctx.restore();
}

// -------- trace picking --------
function hasSameLen(a,b){ return a && b && a.length && b.length && a.length === b.length; }

function buildTraces(frame){
  const wv = frame?.waves;
  if(!wv) return [];

  const v  = wv.vSrc;
  const i  = wv.i;
  const vL = wv.vL;
  const vC = wv.vC;

  const onV  = !!$("#chV")?.checked;
  const onI  = !!$("#chI")?.checked;
  const onVL = !!$("#chVL")?.checked;
  const onVC = !!$("#chVC")?.checked;

  const traces = [];

  if(onV && onI && hasSameLen(v,i))   traces.push({ key:"vSrc_i",  X:v, Y:i,  label:"CH1–CH2 (v vs i)",   baseCol: COL_CH2 });
  if(onV && onVL && hasSameLen(v,vL)) traces.push({ key:"vSrc_vL", X:v, Y:vL, label:"CH1–CH3 (v vs vL)",  baseCol: COL_CH3 });
  if(onV && onVC && hasSameLen(v,vC)) traces.push({ key:"vSrc_vC", X:v, Y:vC, label:"CH1–CH4 (v vs vC)",  baseCol: COL_CH4 });
  if(onI && onVC && hasSameLen(i,vC)) traces.push({ key:"i_vC",    X:i, Y:vC, label:"CH2–CH4 (i vs vC)",  baseCol: mix(COL_CH2, COL_CH4, 0.55), energyCapable:true });

  if(traces.length === 0){
    if(onI && onVL && hasSameLen(i,vL)) traces.push({ key:"vSrc_vL", X:i, Y:vL, label:"CH2–CH3 (i vs vL)", baseCol: COL_CH3 });
    else if(onI && onVC && hasSameLen(i,vC)) traces.push({ key:"i_vC", X:i, Y:vC, label:"CH2–CH4 (i vs vC)", baseCol: COL_CH4, energyCapable:true });
  }

  return traces.slice(0, 4);
}

function updateTraceScale(traceKey, X, Y){
  const st = _scale[traceKey] || (_scale[traceKey] = { fsX:null, fsY:null });

  let fsX = 1e-9, fsY = 1e-9;
  for(let k=0;k<X.length;k++){
    fsX = Math.max(fsX, Math.abs(X[k]));
    fsY = Math.max(fsY, Math.abs(Y[k]));
  }
  fsX *= 1.15;
  fsY *= 1.15;

  const a = 0.12;
  if(st.fsX === null) st.fsX = fsX;
  if(st.fsY === null) st.fsY = fsY;
  st.fsX = st.fsX + a*(fsX - st.fsX);
  st.fsY = st.fsY + a*(fsY - st.fsY);

  return st;
}

// -------- renderer --------
export function renderFieldView(frame) {
  if (!ctx || !canvas || !frame || !tctx || !trail) return;

  const mode = $("#viewMode")?.value || localStorage.getItem("viewMode") || "scope";
  if (mode !== "field") return;

  const traces = buildTraces(frame);
  const W = canvas.width / dpr;
  const H = canvas.height / dpr;

  ctx.setTransform(dpr, 0, 0, dpr, 0, 0);
  tctx.setTransform(dpr, 0, 0, dpr, 0, 0);

  if (!traces.length) {
    ctx.clearRect(0, 0, W, H);
    clearTrails();
    ctx.save();
    ctx.fillStyle = "rgba(230,237,243,0.92)";
    ctx.font = "900 12px ui-sans-serif, system-ui";
    ctx.shadowColor = "rgba(0,0,0,0.55)";
    ctx.shadowBlur = 6;
    ctx.fillText("Field view: enable compatible channels (CH1/CH2/CH3/CH4).", 18, 26);
    ctx.restore();
    return;
  }

  // persistence fade (trail buffer only)
  const persist = persistValue();
  tctx.save();
  tctx.globalCompositeOperation = "destination-out";
  tctx.fillStyle = `rgba(0,0,0,${1 - persist})`;
  tctx.fillRect(0, 0, W, H);
  tctx.restore();

  // redraw background fresh
  ctx.clearRect(0, 0, W, H);

  const pad = 18;
  const x0 = pad, y0 = pad;
  const w = W - 2*pad, h = H - 2*pad;
  const midX = x0 + w/2, midY = y0 + h/2;

  ctx.save();
  ctx.strokeStyle = "rgba(30,42,58,.70)";
  ctx.lineWidth = 1;
  ctx.strokeRect(x0, y0, w, h);

  ctx.strokeStyle = "rgba(151,166,178,.18)";
  ctx.beginPath();
  ctx.moveTo(midX, y0); ctx.lineTo(midX, y0 + h);
  ctx.moveTo(x0, midY); ctx.lineTo(x0 + w, midY);
  ctx.stroke();
  ctx.restore();

  const wv = frame.waves || {};
  const canEnergy = !!(wv.i && wv.vC && wv.i.length && wv.vC.length && wv.i.length === wv.vC.length && (state.C || 0) > 0);
  const headIdxE = canEnergy ? (wv.i.length - 1) : 0;

  const Eraw = canEnergy ? energyTotal(wv.i[headIdxE], wv.vC[headIdxE]) : 0;
  const Bal  = canEnergy ? energyBalance(wv.i[headIdxE], wv.vC[headIdxE]) : 0;

  const now = performance.now() * 1e-3;
  if (lastT === null) lastT = now;
  const dt = Math.max(1e-3, Math.min(0.05, now - lastT));
  lastT = now;

  const kE  = 1 - Math.exp(-dt * 6.0);
  const kdE = 1 - Math.exp(-dt * 10.0);

  E_s = E_s + kE * (Eraw - E_s);
  const dE = (Eraw - lastEraw) / dt;
  dE_s = dE_s + kdE * (dE - dE_s);
  lastEraw = Eraw;

  let E_norm = 0;
  if (canEnergy){
    const x = Math.max(0, E_s);
    E_norm = 1 - Math.exp(-x * 8.0);
    E_norm = Math.max(0, Math.min(1, E_norm));
  }
  const bal_norm = Math.max(0, Math.min(1, Bal));

  let shimmer = 0;
  if (canEnergy){
    const s = Math.min(1, Math.abs(dE_s) * 0.02);
    shimmer = Math.max(0, Math.min(1, s)) * shimmerValue();
  }

  drawEnergyFog(W, H, midX, midY, E_norm, bal_norm, shimmer);

  // projection
  const depth = depthValue();
  const rotSpd = rotValue();
  const tNow = now;

  const rotGain = (0.12 + 0.88 * bal_norm) * (1.0 + 0.25 * shimmer);
  const ang = rotSpd * rotGain * 0.22 * tNow;
  const ca = Math.cos(ang), sa = Math.sin(ang);

  function proj(nx, ny){
    const phase = Math.atan2(ny, nx);
    const z = Math.sin(phase) * (0.75 + 0.25 * bal_norm);

    let px = nx;
    let py = ny;

    const swirl = 0.03 * depth * (0.4 + 0.6 * E_norm);
    const sw = Math.sin(phase * 2.0 + tNow * 0.7) * swirl;
    px += sw * ny;
    py -= sw * nx;

    let pz = z * (0.55 + 0.75 * depth) * (0.85 + 0.35 * shimmer);

    const rx = px * ca - py * sa;
    const ry = px * sa + py * ca;
    const rz = pz;

    const f = 2.15;
    const k = f / (f + rz);

    return {
      x: midX + rx * k * (w * 0.46),
      y: midY - ry * k * (h * 0.46),
      phase,
      z: rz
    };
  }

  // draw into trail buffer
  const glow = glowValue();
  const dot  = dotValue();
  const useEnergyColor = energyColorOn() && canEnergy;

  for(let ti=0; ti<traces.length; ti++){
    const tr = traces[ti];
    const X = tr.X, Y = tr.Y;
    const N = Math.min(X.length, Y.length);
    if(N < 4) continue;

    const step = Math.max(1, Math.floor(N / 1600));
    const sc = updateTraceScale(tr.key, X, Y);
    const sx = 1 / Math.max(1e-12, sc.fsX);
    const sy = 1 / Math.max(1e-12, sc.fsY);

    const baseCol = tr.baseCol || [151,166,178];

    // glow
    tctx.save();
    tctx.globalCompositeOperation = "lighter";
    tctx.lineCap = "round";
    tctx.lineJoin = "round";
    const glowA = (0.06 + 0.10 * glow) * (0.85 - 0.12*ti);
    tctx.strokeStyle = rgba(baseCol, glowA);
    tctx.lineWidth = (2.2 + 3.2 * glow) * (0.90 - 0.10*ti);

    tctx.beginPath();
    let p0 = proj(X[0]*sx, Y[0]*sy);
    tctx.moveTo(p0.x, p0.y);
    for(let k=step; k<N; k+=step){
      const p = proj(X[k]*sx, Y[k]*sy);
      tctx.lineTo(p.x, p.y);
    }
    tctx.stroke();
    tctx.restore();

    // main
    tctx.save();
    tctx.globalCompositeOperation = "lighter";
    tctx.lineCap = "round";
    tctx.lineJoin = "round";

    const baseA = 0.16 * (0.92 - 0.10*ti);
    const baseW = 1.08 * (0.95 - 0.08*ti);

    if(!useEnergyColor){
      tctx.strokeStyle = rgba(baseCol, baseA);
      tctx.lineWidth = baseW;

      tctx.beginPath();
      p0 = proj(X[0]*sx, Y[0]*sy);
      tctx.moveTo(p0.x, p0.y);
      for(let k=step; k<N; k+=step){
        const p = proj(X[k]*sx, Y[k]*sy);
        tctx.lineTo(p.x, p.y);
      }
      tctx.stroke();
    } else {
      const chunk = Math.max(step, Math.floor(80 / Math.max(1, step)) * step);
      for(let s=0; s<N-step; s+=chunk){
        const e = Math.min(N-1, s+chunk);
        const m = Math.min(N-1, s + Math.floor((e-s)/2));

        const tt = energyMixFactor(wv.i[m] || 0, wv.vC[m] || 0);
        const col = mix(COL_CH3, COL_CH4, tt);

        const balHere = energyBalance(wv.i[m] || 0, wv.vC[m] || 0);
        const aHere = baseA * (0.75 + 0.45 * balHere);
        const wHere = baseW * (0.95 + 0.30 * balHere);

        tctx.strokeStyle = rgba(col, aHere);
        tctx.lineWidth = wHere;

        tctx.beginPath();
        let p = proj(X[s]*sx, Y[s]*sy);
        tctx.moveTo(p.x, p.y);
        for(let k=s+step; k<=e; k+=step){
          p = proj(X[k]*sx, Y[k]*sy);
          tctx.lineTo(p.x, p.y);
        }
        tctx.stroke();
      }
    }
    tctx.restore();

    // traveler
    if(dot > 0.01){
      const headIdx = N-1;
      const head = proj(X[headIdx]*sx, Y[headIdx]*sy);

      let headCol = baseCol;
      if(useEnergyColor){
        const tt = energyMixFactor(wv.i[headIdxE] || 0, wv.vC[headIdxE] || 0);
        headCol = mix(COL_CH3, COL_CH4, tt);
      }

      const bloom = (1.0 + 0.9 * E_norm + 0.6 * shimmer) * dot * (1.0 - 0.10*ti);

      tctx.save();
      tctx.globalCompositeOperation = "lighter";
      tctx.shadowColor = rgba(headCol, 0.55);
      tctx.shadowBlur = 18 * bloom;

      tctx.fillStyle = rgba(headCol, 0.92);
      tctx.beginPath();
      tctx.arc(head.x, head.y, 3.2 * bloom, 0, Math.PI * 2);
      tctx.fill();

      tctx.shadowBlur = 0;
      tctx.fillStyle = "rgba(255,255,255,0.74)";
      tctx.beginPath();
      tctx.arc(head.x, head.y, 1.15 * bloom, 0, Math.PI * 2);
      tctx.fill();

      tctx.restore();
    }
  }

  // composite trails
  ctx.save();
  ctx.globalCompositeOperation = "lighter";
  ctx.drawImage(trail, 0, 0, W, H);
  ctx.restore();

  // labels (FIELD legend bottom-left)
  ctx.save();

  const legendPad = 12;
  const legendLineH = 18;

  const title = "FIELD VIEW · MULTI-CHANNEL ENERGY CHAMBER";
  const traceLine = traces.map(t => t.label).join("  ·  ");
  const persistLine = `persist=${persist.toFixed(2)}`;

  // sizing
  ctx.font = "12px system-ui, sans-serif";
  const w1 = ctx.measureText(title).width;
  const w2 = ctx.measureText(traceLine).width;
  const w3 = ctx.measureText(persistLine).width;

  const boxW = Math.max(w1, w2, w3) + legendPad * 2;
  const boxH = legendLineH * 3 + legendPad * 2;

  const boxX = x0 + 10;
  const boxY = y0 + h - boxH - 10;

  // background
  ctx.globalAlpha = 0.65;
  ctx.fillStyle = "rgba(12,18,26,0.85)";
  ctx.fillRect(boxX, boxY, boxW, boxH);

  // text
  ctx.globalAlpha = 1;
  ctx.fillStyle = "rgba(230,237,243,0.95)";
  ctx.fillText(title,       boxX + legendPad, boxY + legendPad + legendLineH * 0.8);
  ctx.fillText(traceLine,   boxX + legendPad, boxY + legendPad + legendLineH * 1.8);
  ctx.fillText(persistLine, boxX + legendPad, boxY + legendPad + legendLineH * 2.8);

  ctx.restore();
}