// export_image.js
// INSANE hi-res PNG export using TILED rendering (avoids giant-canvas GPU/memory limits).
// Exports the *currently visible* canvas (scope/field/helix) + embeds ALL user inputs inside the PNG (tEXt chunk).
// NEW: also writes a sidecar .json file containing the same snapshot for traceability.
// NEW: bg=2 -> transparent PNG export (alpha preserved).
// CINEMATIC PRO: optional tile-level unsharp mask to restore crispness when upscaling.

import { state } from "./state.js";
import { Rtotal, tau, periodPulse, periodSine, duty } from "./core/derived.js";

function isVisible(el){
  if(!el) return false;
  const cs = getComputedStyle(el);
  if(cs.display === "none" || cs.visibility === "hidden" || cs.opacity === "0") return false;
  const r = el.getBoundingClientRect();
  return (r.width > 10 && r.height > 10);
}

function pickVisibleCanvas({ scopeCanvas, fieldCanvas, helixCanvas }){
  if(isVisible(helixCanvas)) return { canvas: helixCanvas, mode: "helix" };
  if(isVisible(fieldCanvas)) return { canvas: fieldCanvas, mode: "field" };
  return { canvas: scopeCanvas, mode: "scope" };
}

function safeNowStamp(){
  const d = new Date();
  const pad = (n)=> String(n).padStart(2,"0");
  return `${d.getFullYear()}-${pad(d.getMonth()+1)}-${pad(d.getDate())}_${pad(d.getHours())}${pad(d.getMinutes())}${pad(d.getSeconds())}`;
}

function downloadBlob(blob, filename){
  const url = URL.createObjectURL(blob);
  const a = document.createElement("a");
  a.href = url;
  a.download = filename;
  document.body.appendChild(a);
  a.click();
  a.remove();
  setTimeout(()=> URL.revokeObjectURL(url), 1500);
}

function downloadText(text, filename, mime="application/json"){
  const blob = new Blob([text], { type: mime });
  downloadBlob(blob, filename);
}

function clamp(v, a, b){ return Math.max(a, Math.min(b, v)); }

function drawBackground(ctx, W, H, stage, bgMode){
  // bgMode:
  //  - "video"       : draw video + subtle overlay
  //  - "dark"|"off"  : solid dark + subtle overlay
  //  - "transparent" : DO NOTHING (keep alpha)
  if(bgMode === "transparent") return;

  // base fill
  ctx.fillStyle = "#05070b";
  ctx.fillRect(0, 0, W, H);

  if(bgMode === "video"){
    const video = stage?.querySelector?.("video");
    if(video && video.readyState >= 2){
      try { ctx.drawImage(video, 0, 0, W, H); } catch(_) {}
    }
  }

  // subtle overlay
  ctx.fillStyle = "rgba(0,0,0,0.18)";
  ctx.fillRect(0, 0, W, H);
}

async function yieldToUI(){
  await new Promise(r => setTimeout(r, 0));
}

// ---------- PNG metadata embedding (VALID) ----------
// We embed payload into a standard PNG tEXt chunk:
//   data = keyword + NUL + text
// and we insert the chunk AFTER IHDR.

function crc32(buf){
  let table = crc32.table;
  if(!table){
    table = crc32.table = [];
    for(let i=0;i<256;i++){
      let c = i;
      for(let k=0;k<8;k++){
        c = ((c & 1) ? (0xEDB88320 ^ (c >>> 1)) : (c >>> 1));
      }
      table[i] = c;
    }
  }
  let crc = -1;
  for(let i=0;i<buf.length;i++){
    crc = (crc >>> 8) ^ table[(crc ^ buf[i]) & 0xFF];
  }
  return (crc ^ (-1)) >>> 0;
}

function makeTextChunk(keyword, text){
  const enc = new TextEncoder();
  const type = enc.encode("tEXt");
  const data = enc.encode(String(keyword) + "\0" + String(text));

  const lenBytes = new Uint8Array(4);
  new DataView(lenBytes.buffer).setUint32(0, data.length);

  const crcInput = new Uint8Array(type.length + data.length);
  crcInput.set(type, 0);
  crcInput.set(data, type.length);

  const crcVal = crc32(crcInput);
  const crcBytes = new Uint8Array(4);
  new DataView(crcBytes.buffer).setUint32(0, crcVal);

  const chunk = new Uint8Array(4 + 4 + data.length + 4);
  chunk.set(lenBytes, 0);
  chunk.set(type, 4);
  chunk.set(data, 8);
  chunk.set(crcBytes, 8 + data.length);
  return chunk;
}

async function embedPngText(blob, keyword, text){
  const buf = await blob.arrayBuffer();
  const bytes = new Uint8Array(buf);

  // find position after IHDR chunk
  let offset = 8; // signature
  let insertPos = null;

  while(offset + 8 <= bytes.length){
    const len = new DataView(buf).getUint32(offset);
    const type = String.fromCharCode(
      bytes[offset+4], bytes[offset+5], bytes[offset+6], bytes[offset+7]
    );
    const chunkTotal = 12 + len; // len + type + data + crc
    if(type === "IHDR"){
      insertPos = offset + chunkTotal;
      break;
    }
    offset += chunkTotal;
    if(offset > bytes.length) break;
  }

  if(insertPos == null){
    // fallback: just return original
    return blob;
  }

  const chunk = makeTextChunk(keyword, text);

  const out = new Uint8Array(bytes.length + chunk.length);
  out.set(bytes.slice(0, insertPos), 0);
  out.set(chunk, insertPos);
  out.set(bytes.slice(insertPos), insertPos + chunk.length);

  return new Blob([out], { type: "image/png" });
}

// ---------- portability helpers (ASCII-only payload) ----------
function utf8ToB64(str){
  const u8 = new TextEncoder().encode(str);
  let bin = "";
  const CHUNK = 0x8000;
  for(let i=0;i<u8.length;i+=CHUNK){
    bin += String.fromCharCode.apply(null, u8.subarray(i, i+CHUNK));
  }
  return btoa(bin);
}

function safeClone(obj){
  try { return JSON.parse(JSON.stringify(obj)); } catch(_) { return null; }
}

function collectAllControls(root = document){
  const out = [];
  const els = root.querySelectorAll("input, select, textarea");
  for(const el of els){
    const id = el.id || el.name || null;
    if(!id) continue;

    const tag = el.tagName.toLowerCase();
    const item = { id, tag };

    if(tag === "select"){
      item.value = el.value;
      item.selectedIndex = el.selectedIndex;
      item.options = [];
      for(const opt of el.options){
        item.options.push({ v: opt.value, s: opt.selected });
      }
      out.push(item);
      continue;
    }

    const type = (el.type || "").toLowerCase();
    item.type = type;

    if(type === "checkbox" || type === "radio"){
      item.checked = !!el.checked;
      item.value = el.value ?? null;
      out.push(item);
      continue;
    }

    item.value = el.value ?? null;
    if(el.min != null && el.min !== "") item.min = el.min;
    if(el.max != null && el.max !== "") item.max = el.max;
    if(el.step != null && el.step !== "") item.step = el.step;

    out.push(item);
  }
  return out;
}

// ---------- CINEMATIC PRO: fast unsharp mask (radius=1) ----------
function unsharpTile(ctx, w, h, amount = 0.35){
  // amount 0.25..0.55 is useful; keep it subtle for “cinematic”
  const img = ctx.getImageData(0, 0, w, h);
  const src = img.data;
  const tmp = new Uint8ClampedArray(src.length);
  const blur = new Uint8ClampedArray(src.length);

  // horizontal box blur radius=1 (3 taps)
  for(let y=0; y<h; y++){
    const row = y * w * 4;
    for(let x=0; x<w; x++){
      const xm1 = Math.max(0, x-1);
      const xp1 = Math.min(w-1, x+1);

      const a = row + xm1*4;
      const b = row + x*4;
      const c = row + xp1*4;

      const o = row + x*4;
      tmp[o]   = (src[a]   + src[b]   + src[c])   / 3;
      tmp[o+1] = (src[a+1] + src[b+1] + src[c+1]) / 3;
      tmp[o+2] = (src[a+2] + src[b+2] + src[c+2]) / 3;
      tmp[o+3] = src[o+3];
    }
  }

  // vertical box blur radius=1 (3 taps)
  for(let y=0; y<h; y++){
    const ym1 = Math.max(0, y-1);
    const yp1 = Math.min(h-1, y+1);
    for(let x=0; x<w; x++){
      const a = (ym1*w + x)*4;
      const b = (y*w   + x)*4;
      const c = (yp1*w + x)*4;

      blur[b]   = (tmp[a]   + tmp[b]   + tmp[c])   / 3;
      blur[b+1] = (tmp[a+1] + tmp[b+1] + tmp[c+1]) / 3;
      blur[b+2] = (tmp[a+2] + tmp[b+2] + tmp[c+2]) / 3;
      blur[b+3] = tmp[b+3];
    }
  }

  // unsharp: out = src + amount*(src - blur)
  const k = amount;
  for(let i=0; i<src.length; i+=4){
    const r = src[i]   + k*(src[i]   - blur[i]);
    const g = src[i+1] + k*(src[i+1] - blur[i+1]);
    const b = src[i+2] + k*(src[i+2] - blur[i+2]);

    src[i]   = r < 0 ? 0 : (r > 255 ? 255 : r);
    src[i+1] = g < 0 ? 0 : (g > 255 ? 255 : g);
    src[i+2] = b < 0 ? 0 : (b > 255 ? 255 : b);
    // alpha unchanged
  }

  ctx.putImageData(img, 0, 0);
}

export function initImageExport(opts = {}){
  const {
    stageSel = ".stageTop",
    scopeCanvas,
    fieldCanvas,
    helixCanvas,
    viewSel = null,
    bgModeSel = "#bgMode"
  } = opts;

  const btn = document.querySelector("#saveImg");
  if(!btn) return false;

  btn.addEventListener("click", async () => {
    // --- snapshot: freeze animation ONLY during tiled capture ---
    const prevHold = !!state.hold;
    state.hold = true;

    // let one frame settle so the canvas is stable before we start tiling
    await yieldToUI();
    await yieldToUI();
    const stage = document.querySelector(stageSel) || scopeCanvas?.parentElement;
    if(!stage || !scopeCanvas) return;

    // =========================
    // User settings
    // =========================
    let scale = 8;         // default insane
    let tilePx = 2048;     // safe default
    let bg = 1;            // 1=normal bg (video/dark selector), 0=dark-only, 2=TRANSPARENT PNG
    let sharp = 1;         // CINEMATIC PRO default ON

    const raw = prompt(
      "PNG Export Settings:\n\n" +
      "Format:\n" +
      "  scale , tilePx , bg , sharp\n\n" +
      "Parameters:\n" +
      "  scale   = output multiplier (1–32)\n" +
      "  tilePx  = tile size in px (512–4096)\n" +
      "  bg      = 1 normal  |  0 dark-only  |  2 transparent\n" +
      "  sharp   = 1 enabled |  0 disabled\n\n" +
      "Examples:\n" +
      "  6,2048,1,1   → balanced high quality\n" +
      "  8,2048,1,1   → very high detail\n" +
      "  12,1536,0,1  → extreme resolution, dark background\n" +
      "  8,2048,2,1   → transparent PNG\n" +
      "  8,2048,1,0   → no sharpening\n",
      "8,2048,1,1"
    );

    if(raw != null){
      const parts = String(raw).split(",").map(s => s.trim());
      const s  = parseFloat(parts[0]?.replace(",", "."));
      const t  = parseInt(parts[1] || "2048", 10);
      const b  = parseInt(parts[2] || "1", 10);
      const sh = parseInt(parts[3] || "1", 10);

      if(isFinite(s)) scale = clamp(s, 1, 32);
      if(isFinite(t)) tilePx = clamp(t, 512, 4096);

      // bg: clamp to {0,1,2}
      bg = (b === 2) ? 2 : (b === 0 ? 0 : 1);

      sharp = (sh !== 0) ? 1 : 0;
    }

    const bgMode =
      (bg === 2) ? "transparent" :
      (bg === 1) ? (document.querySelector(bgModeSel)?.value || "video") :
      "off";

    const rect = stage.getBoundingClientRect();
    const outW = Math.max(1, Math.floor(rect.width  * scale));
    const outH = Math.max(1, Math.floor(rect.height * scale));

    // Choose visible canvas
    const pick = pickVisibleCanvas({ scopeCanvas, fieldCanvas, helixCanvas });

    // Output canvas
    const out = document.createElement("canvas");
    out.width = outW;
    out.height = outH;

    // IMPORTANT: alpha must be TRUE for transparent exports
    const outCtx = out.getContext("2d", { alpha: (bg === 2) });
    outCtx.imageSmoothingEnabled = true;
    try { outCtx.imageSmoothingQuality = "high"; } catch(_) {}

    // Draw background once (or skip if transparent)
    drawBackground(outCtx, outW, outH, stage, bgMode);

    // Tile canvas
    const tile = document.createElement("canvas");
    tile.width = tilePx;
    tile.height = tilePx;

    const tctx = tile.getContext("2d", { alpha: true, desynchronized: true, willReadFrequently: true });
    tctx.imageSmoothingEnabled = true;
    try { tctx.imageSmoothingQuality = "high"; } catch(_) {}

    // Crop mapping: output px -> css px -> source px
    let src = pick.canvas;

    // If this view uses an offscreen persistence buffer, composite it for export.
    const trailCanvas = src && src.__trailCanvas ? src.__trailCanvas : null;
    if(trailCanvas && (pick.mode === "field" || pick.mode === "helix")){
      const comp = document.createElement("canvas");
      comp.width = src.width;
      comp.height = src.height;

      const cctx = comp.getContext("2d", { alpha: true, desynchronized: true });
      // trails first, then UI/head on top
      try { cctx.drawImage(trailCanvas, 0, 0); } catch(_){}
      try { cctx.drawImage(src, 0, 0); } catch(_){}

      src = comp;
    }

    const sxScale = src.width  / rect.width;
    const syScale = src.height / rect.height;

    const tilesX = Math.ceil(outW / tilePx);
    const tilesY = Math.ceil(outH / tilePx);

    btn.disabled = true;
    const oldText = btn.textContent;
    btn.textContent = `Exporting… 0%`;

    let done = 0;
    const total = tilesX * tilesY;

    for(let ty = 0; ty < tilesY; ty++){
      for(let tx = 0; tx < tilesX; tx++){
        const dx = tx * tilePx;
        const dy = ty * tilePx;
        const dw = Math.min(tilePx, outW - dx);
        const dh = Math.min(tilePx, outH - dy);

        tctx.clearRect(0, 0, tilePx, tilePx);

        const cssX = dx / scale;
        const cssY = dy / scale;
        const cssW = dw / scale;
        const cssH = dh / scale;

        const sx = cssX * sxScale;
        const sy = cssY * syScale;
        const sw = cssW * sxScale;
        const sh = cssH * syScale;

        try{
          // upscale into tile
          tctx.drawImage(src, sx, sy, sw, sh, 0, 0, dw, dh);

          // CINEMATIC PRO: sharpen the *upscaled* tile
          if(sharp){
            // amount tuned for “glow + crisp core”
            unsharpTile(tctx, dw, dh, 0.38);
          }

          // stamp onto final
          outCtx.drawImage(tile, 0, 0, dw, dh, dx, dy, dw, dh);
        }catch(_){}

        done++;
        if(done % 3 === 0){
          const pct = Math.floor((done / total) * 100);
          btn.textContent = `Exporting… ${pct}%`;
          await yieldToUI();
        }
      }
    }

    // Build FULL metadata payload (all user input + state + derived)
    const timestamp = safeNowStamp();
    const viewVal = viewSel?.value || null;

    const meta = {
      schema: "RLCAnalyzer.Export",
      schemaVersion: 2,
      timestamp,
      mode: pick.mode,
      view: viewVal,
      export: {
        scale,
        tilePx,
        bg,       // 0/1/2 (2=transparent)
        bgMode,   // "video"|"off"|"transparent"
        sharp: !!sharp,
      },
      stageCssPx: { w: rect.width, h: rect.height },
      state: safeClone(state),
      derived: {
        Rtotal: Rtotal(),
        tau: tau(),
        periodPulse: periodPulse(),
        periodSine: periodSine(),
        duty: duty(),
      },
      controls: collectAllControls(),
    };

    const base = `rlc_analyzer_${pick.mode}_${timestamp}_x${scale.toFixed(2)}` +
                 (bg === 2 ? "_transparent" : "");
    const pngName  = `${base}.png`;
    const jsonName = `${base}.json`;

    out.toBlob(async (blob) => {
      btn.disabled = false;
      btn.textContent = oldText;

      // restore previous hold state after capture
      state.hold = prevHold;

      if(!blob) return;

      // 1) Sidecar JSON (readable)
      downloadText(JSON.stringify(meta, null, 2), jsonName, "application/json");

      // 2) PNG with embedded snapshot (compact)
      const payload = "b64:" + utf8ToB64(JSON.stringify(meta));
      const enriched = await embedPngText(blob, "RLCAnalyzer", payload);
      downloadBlob(enriched, pngName);
    }, "image/png");
  });

  return true;
}